/**
 ******************************************************************************
 * @file    main.c
 * @author  Milandr Application Team
 * @version V1.0.1
 * @date    05/12/2025
 * @brief   This example shows how to use MILx.
 ******************************************************************************
 * THE PRESENT FIRMWARE IS FOR GUIDANCE ONLY. IT AIMS AT PROVIDING CUSTOMERS
 * WITH CODING INFORMATION REGARDING MILANDR'S PRODUCTS IN ORDER TO FACILITATE
 * THE USE AND SAVE TIME. MILANDR SHALL NOT BE HELD LIABLE FOR ANY
 * DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES RESULTING
 * FROM THE CONTENT OF SUCH FIRMWARE AND/OR A USE MADE BY CUSTOMERS OF THE
 * CODING INFORMATION CONTAINED HEREIN IN THEIR PRODUCTS.
 */

/* Includes ------------------------------------------------------------------*/
#include "MDR1986VE8T.h"
#include "spec.h"

#include "mdr32f8_config.h"
#include "mdr32f8_port.h"
#include "mdr32f8_clkctrl.h"
#include "mdr32f8_milstd.h"

/* Private function prototypes -----------------------------------------------*/
void ClockConfig(void);
void LedConfig(void);

void MIL1_Config(void);
void MIL2_Config(void);
void MIL1_PinConfig(void);
void MIL2_PinConfig(void);

void MILx_StartTransmission(MIL1553Control* MIL_STD_1553x);

/* Private functions ---------------------------------------------------------*/
void Delay(__IO uint32_t nCount)
{
    for (; nCount != 0; nCount--) { }
}

/**
 * @brief  Main program.
 * @param  None
 * @retval None
 */
int main(void)
{
    ClockConfig();
    LedConfig();

    MIL1_Config(); // Remote Terminal
    MIL1_PinConfig();

    MIL2_Config(); // Bus Controller
    MIL2_PinConfig();

    /* Enable MIL_STD_15531 and MIL_STD_15532 */
    MIL_STD_1553_Cmd(MIL_STD_15531, ENABLE);
    MIL_STD_1553_Cmd(MIL_STD_15532, ENABLE);

    while (1) {
        MILx_StartTransmission(MIL_STD_15532);

        while ((MIL_STD_1553_BCSTART_bit_stat(MIL_STD_15532)) != RESET) { } // Wait for transmission to complete

        if (MIL_STD_1553_GetFlagStatus(MIL_STD_15532, MIL_STD_1553_FLAG_VALMESS) != RESET) {
            PORT_SetBits(PORTC, PORT_Pin_16);   // Set VD7
            PORT_ResetBits(PORTC, PORT_Pin_23); // Reset VD14
        }
        if (MIL_STD_1553_GetFlagStatus(MIL_STD_15532, MIL_STD_1553_FLAG_ERR) != RESET) {
            PORT_SetBits(PORTC, PORT_Pin_23);   // Set VD14
            PORT_ResetBits(PORTC, PORT_Pin_16); // Reset VD7
        }

        Delay(200000);
        PORT_ResetBits(PORTC, PORT_Pin_16 | PORT_Pin_23); // Reset VD7 and VD14
        Delay(200000);
    }
}

void MILx_StartTransmission(MIL1553Control* MIL_STD_1553x)
{
    MIL_STD_1553_CommandWordTypeDef MIL_STD_1553_CommandWord1 = { 0 };

    /* Generate a command word to send */
    MIL_STD_1553_CommandWord1.Fields.Data                  = 5; // Number of data words being received or transmitted
    MIL_STD_1553_CommandWord1.Fields.Subaddress            = 1; // Subaddress where received or transmitted data words are located in memory
    MIL_STD_1553_CommandWord1.Fields.ReadWriteBit          = 0; // Receive/transmit bit (data transfer): 0 - from BC to the RT, 1 - from RT to BC
    MIL_STD_1553_CommandWord1.Fields.TerminalDeviceAddress = 1; // Set RT address

    /* Write the command word to the buffer */
    MIL_STD_1553_SetCommandWord(MIL_STD_1553x, MIL_STD_1553_COMMAND_WORD1, &MIL_STD_1553_CommandWord1);
    /* Set the active transmission channel to the primary channel */
    MIL_STD_1553_Transmitter_CMD(MIL_STD_1553x, MIL_STD_1553_TRANSMITTER_MAIN, ENABLE);
    /* Start transmission */
    MIL_STD_1553_StartTransmision(MIL_STD_1553x);
}

void ClockConfig(void)
{
    UNLOCK_UNIT(CLK_CNTR_key);
    UNLOCK_UNIT(BKP_key);

    CLKCTRL_DeInit();

    /* Enable HSE0 clock */
    CLKCTRL_HSEconfig(CLKCTRL_HSE0_CLK_ON);
    while (CLKCTRL_HSEstatus(CLKCTRL_HSEn_STAT_HSE0_RDY) != SUCCESS) { }

    /* Select PLL0 clk src, PLL0_N, PLL0_Q to get FINT = FIN*(PLLn_N)/(PLLn_Q+1) */
    CLKCTRL_CPU_PLLconfig(PLL0, CLKCTRL_PLLn_CLK_SELECT_HSE0div1, PLL_DIVQ_Q_1_DV, 8); // PLLn, SRC, Q, DV, N
    while (CLKCTRL_CPU_PLLstatus(0) != SUCCESS) { }

    BKPCNTR_SRILOWconfig(BKPCNTR_SRILOW_upto_50MHz);
    CLKCTRL_MAX_CLKSelection(CLKCTRL_MAX_CLK_PLL0);
}

void LedConfig(void)
{
    PORT_InitTypeDef PORT_InitStructure;

    /* Enable PORTC clk */
    CLKCTRL_PER0_CLKcmd(CLKCTRL_PER0_CLK_MDR_PORTC_EN, ENABLE);

    UNLOCK_UNIT(PORTC_key);

    /* Configure PORTC for LED, PC[16:23] */
    PORT_StructInit(&PORT_InitStructure);
    PORT_InitStructure.PORT_Pin     = (PORT_Pin_16 | PORT_Pin_17 | PORT_Pin_18 | PORT_Pin_19 |
                                       PORT_Pin_20 | PORT_Pin_21 | PORT_Pin_22 | PORT_Pin_23);
    PORT_InitStructure.PORT_SOE     = PORT_SOE_OUT;
    PORT_InitStructure.PORT_SANALOG = PORT_SANALOG_DIGITAL;
    PORT_InitStructure.PORT_SPD     = PORT_SPD_OFF;
    PORT_InitStructure.PORT_SPWR    = PORT_SPWR_10;

    PORT_Init(PORTC, &PORT_InitStructure);
}

void MIL1_Config(void)
{
    MIL_STD_1553_InitTypeDef MIL_STD_1553_InitStructure;

    CLKCTRL_PER1_CLKcmd(CLKCTRL_PER1_CLK_MDR_MIL0_EN, ENABLE);

    /*-- Configure remote terminal -----------------------------------------*/
    MIL_STD_1553xStructInit(&MIL_STD_1553_InitStructure);

    /* Init MIL_STD_1553_InitStructure */
    MIL_STD_1553_InitStructure.MIL_STD_1553_Mode = MIL_STD_1553_ModeTerminal;
    MIL_STD_1553_InitStructure.MIL_STD_1553_RERR = DISABLE;
    MIL_STD_1553_InitStructure.MIL_STD_1553_DIV  = 40;
    MIL_STD_1553_InitStructure.MIL_STD_1553_RTA  = 0x01;
    MIL_STD_1553_InitStructure.MIL_STD_1553_TRA  = ENABLE;
    MIL_STD_1553_InitStructure.MIL_STD_1553_TRB  = ENABLE;

    /* Configure MIL_STD_15531 */
    MIL_STD_1553_Init(MIL_STD_15531, &MIL_STD_1553_InitStructure);
}

void MIL2_Config(void)
{
    MIL_STD_1553_InitTypeDef MIL_STD_1553_InitStructure;

    CLKCTRL_PER1_CLKcmd(CLKCTRL_PER1_CLK_MDR_MIL1_EN, ENABLE);

    /*-- Configure bus controller -----------------------------------------*/
    MIL_STD_1553xStructInit(&MIL_STD_1553_InitStructure);

    /* Init MIL_STD_1553_InitStructure */
    MIL_STD_1553_InitStructure.MIL_STD_1553_Mode = MIL_STD_1553_ModeBusController;
    MIL_STD_1553_InitStructure.MIL_STD_1553_RERR = ENABLE;
    MIL_STD_1553_InitStructure.MIL_STD_1553_DIV  = 40;
    MIL_STD_1553_InitStructure.MIL_STD_1553_RTA  = 0;
    MIL_STD_1553_InitStructure.MIL_STD_1553_TRA  = ENABLE;
    MIL_STD_1553_InitStructure.MIL_STD_1553_TRB  = DISABLE;

    /* Configure MIL_STD_15532 */
    MIL_STD_1553_Init(MIL_STD_15532, &MIL_STD_1553_InitStructure);
}

void MIL1_PinConfig(void)
{
    PORT_InitTypeDef PORT_InitStructure;

    CLKCTRL_PER0_CLKcmd(CLKCTRL_PER0_CLK_MDR_PORTA_EN, ENABLE);
    CLKCTRL_PER0_CLKcmd(CLKCTRL_PER0_CLK_MDR_PORTB_EN, ENABLE);

    UNLOCK_UNIT(PORTA_key);
    UNLOCK_UNIT(PORTB_key);

    /*****************************XP22*****************************/
    PORT_StructInit(&PORT_InitStructure);

    PORT_InitStructure.PORT_Pin     = (PORT_Pin_1 | PORT_Pin_2);
    PORT_InitStructure.PORT_SFUNC   = PORT_SFUNC_13;
    PORT_InitStructure.PORT_SANALOG = PORT_SANALOG_DIGITAL;
    PORT_InitStructure.PORT_SPWR    = PORT_SPWR_10;
    PORT_Init(PORTB, &PORT_InitStructure);

    PORT_InitStructure.PORT_Pin     = (PORT_Pin_25 | PORT_Pin_26 | PORT_Pin_27 | PORT_Pin_28 | PORT_Pin_29);
    PORT_InitStructure.PORT_SFUNC   = PORT_SFUNC_12;
    PORT_InitStructure.PORT_SANALOG = PORT_SANALOG_DIGITAL;
    PORT_InitStructure.PORT_SPWR    = PORT_SPWR_10;
    PORT_Init(PORTA, &PORT_InitStructure);

    PORT_InitStructure.PORT_Pin     = (PORT_Pin_22);
    PORT_InitStructure.PORT_SFUNC   = PORT_SFUNC_12;
    PORT_InitStructure.PORT_SANALOG = PORT_SANALOG_DIGITAL;
    PORT_InitStructure.PORT_SPWR    = PORT_SPWR_10;
    PORT_Init(PORTA, &PORT_InitStructure);

    PORT_InitStructure.PORT_Pin     = (PORT_Pin_30 | PORT_Pin_31);
    PORT_InitStructure.PORT_SFUNC   = PORT_SFUNC_13;
    PORT_InitStructure.PORT_SANALOG = PORT_SANALOG_DIGITAL;
    PORT_InitStructure.PORT_SPWR    = PORT_SPWR_10;
    PORT_Init(PORTA, &PORT_InitStructure);
}

void MIL2_PinConfig(void)
{
    PORT_InitTypeDef PORT_InitStructure;

    CLKCTRL_PER0_CLKcmd(CLKCTRL_PER0_CLK_MDR_PORTB_EN, ENABLE);
    CLKCTRL_PER0_CLKcmd(CLKCTRL_PER0_CLK_MDR_PORTC_EN, ENABLE);
    CLKCTRL_PER0_CLKcmd(CLKCTRL_PER0_CLK_MDR_PORTD_EN, ENABLE);

    UNLOCK_UNIT(PORTB_key);
    UNLOCK_UNIT(PORTC_key);
    UNLOCK_UNIT(PORTD_key);

    /*****************************XP23*****************************/
    PORT_StructInit(&PORT_InitStructure);

    PORT_InitStructure.PORT_Pin     = (PORT_Pin_28 | PORT_Pin_29 | PORT_Pin_30);
    PORT_InitStructure.PORT_SFUNC   = PORT_SFUNC_13;
    PORT_InitStructure.PORT_SANALOG = PORT_SANALOG_DIGITAL;
    PORT_InitStructure.PORT_SPWR    = PORT_SPWR_10;
    PORT_Init(PORTB, &PORT_InitStructure);

    PORT_InitStructure.PORT_Pin     = (PORT_Pin_1 | PORT_Pin_2 | PORT_Pin_3 | PORT_Pin_4 | PORT_Pin_5);
    PORT_InitStructure.PORT_SFUNC   = PORT_SFUNC_13;
    PORT_InitStructure.PORT_SANALOG = PORT_SANALOG_DIGITAL;
    PORT_InitStructure.PORT_SPWR    = PORT_SPWR_10;
    PORT_Init(PORTC, &PORT_InitStructure);

    PORT_InitStructure.PORT_Pin     = (PORT_Pin_27 | PORT_Pin_28);
    PORT_InitStructure.PORT_SFUNC   = PORT_SFUNC_13;
    PORT_InitStructure.PORT_SANALOG = PORT_SANALOG_DIGITAL;
    PORT_InitStructure.PORT_SPWR    = PORT_SPWR_10;
    PORT_Init(PORTD, &PORT_InitStructure);
}

/*********************** (C) COPYRIGHT 2025 Milandr ****************************
 *
 * END OF FILE main.c */
