#include "tlv320aic23b.h"
#include <sysreg.h>
#include <math.h>
#include <stdio.h>
#include "hal_audio.h"

/* --------------------------------- Список регистров устройства --------------------------------- */
#define TLV_REG_LLIV 						0x00													// Громкость левого линейного входа (+12dB..-34.5dB)
#define TLV_REG_RLIV 						0x01													// Громкость правого линейного входа (+12dB..-34.5dB)
#define TLV_REG_LCHV 						0x02													// Громкость левого канала наушников
#define TLV_REG_RCHV 						0x03													// Громкость левого канала наушников
#define TLV_REG_AAPC 						0x04													// Настройка аналоговой аудио части
#define TLV_REG_DAPC 						0x05													// Настройка цифровой аудио части
#define TLV_REG_PWDC 						0x06													// Управление питанием
#define TLV_REG_DAIF 						0x07													// Формат цифрового аудио интерфейса
#define TLV_REG_SRC 						0x08													// Контроль частоты выборки
#define TLV_REG_DIA 						0x09													// Включение цифрового интерфейса
#define TLV_REG_RESET 						0x0F													// Регист сброса

/* ---------------------------------- Громкость линейного входа --------------------------------- */
#define TLV_RLLIV_LRS         				0x0100													// Одновременное обновление громкости левого/правого канала линейного входа
#define TLV_RLLIV_MUTED           			0x0080													// Выключение канала линейного входа
#define TLV_RLLIV_MAX             			0x001f													// Максимальная громкость +12 dB
#define TLV_RLLIV_MIN             			0x0000													// Минимальная громкость -34.5 dB
/* -------------------------------- Громкость выхода на наушники -------------------------------- */
#define TLV_RLCHV_LRS         				0x0100													// Одновременное обновление громкости левого/правого канала выхода на наушники
#define TLV_RLCHV_ZC_ON              		0x0080													// Обновление громкости в момент пересечения ноля
#define TLV_RLCHV_MAX             			0x007f													// Максимальная громкость +6 dB
#define TLV_RLCHV_MIN             			0x0030													// Минимальная громкость -73 dB
/* ------------------------------ Настройка аналоговой аудио части ------------------------------ */
#define STA_0dB								0x04
#define STA_m6dB							0x00
#define STA_m9dB							0x01
#define STA_m12dB							0x02
#define STA_m18dB							0x03
#define TLV_AAPC_STA_REG(x)          		((x)<<6)												// Громкость самопрослушивания
#define TLV_AAPC_STE_ENABLED         		0x0020													// Включение самопрослушивания
#define TLV_AAPC_DAC_SELECTED        		0x0010													// DAC включен
#define TLV_AAPC_BYPASS_ON           		0x0008													// Включение байпаса с линейного входа на выход науников
#define TLV_AAPC_INSEL_MIC           		0x0004													// Вход ADC - микрофон, иначе линейный вход
#define TLV_AAPC_MIC_MUTED          		0x0002													// Микрофон отключен
#define TLV_AAPC_MIC_20DB           		0x0001													// Усиление микрофона +20 dB
/* ------------------------------- Настройка цифровой аудио части ------------------------------- */
#define TLV_DAPC_DACM_MUTE           		0x0008													// Плавное выключение DAC
#define TLV_DAPC_DEEMP_32K           		0x0002													// Фильтр 32 кгЦ
#define TLV_DAPC_DEEMP_44K           		0x0004													// Фильтр 44.1 кГц
#define TLV_DAPC_DEEMP_48K           		0x0006													// Фильтр 48 кГц
#define TLV_DAPC_ADCHP_OFF            		0x0001													// Выключение высокочастотного фильтра ADC
/* ------------------------------------ Управление питанием ------------------------------------- */
#define TLV_PWDC_DEVICE_PWR_OFF      		0x0080													// Выключение питания устройства
#define TLV_PWDC_CLK_OFF             		0x0040													// Выключение клоков
#define TLV_PWDC_OSC_OFF             		0x0020													// Выключение осциллятора
#define TLV_PWDC_OUT_OFF             		0x0010													// Выключение выхода
#define TLV_PWDC_DAC_OFF             		0x0008													// Выключение DAC
#define TLV_PWDC_ADC_OFF             		0x0004													// Выключение ADC
#define TLV_PWDC_MIC_OFF             		0x0002													// Выключение микрофона
#define TLV_PWDC_LINE_OFF            		0x0001													// Выключение линейного входа
/* -------------------------------- Управление частотой выборки --------------------------------- */
#define SRC_USB_ADC94_DAC96					0x001D
#define SRC_USB_ADC88d2_DAC88d2				0x001F
#define SRC_USB_ADC48_DAC48					0x0001
#define SRC_USB_ADC44d1_DAC44d1				0x0023
#define SRC_USB_ADC32_DAC32					0x0019
#define SRC_USB_ADC8d021_DAC8d021			0x002F
#define SRC_USB_ADC8_DAC8					0x000D
#define SRC_USB_ADC48_DAC8					0x0005
#define SRC_USB_ADC44d1_DAC8d021			0x0027
#define SRC_USB_ADC8_DAC48					0x0009
#define SRC_USB_ADC8d21_DAC44d1				0x002B
#define TLV_SRC_CLKOUT_HALF         		0x0080													// Делитель выходной частоты пополам
#define TLV_SRC_CLKIN_HALF          		0x0040													// Делитель входной частоты пополам
#define TLV_SRC_USB_CLK_ON          		0x0001													// Включение режима тактирования USB
/* ------------------------------- Включение цифрового интерфейса ------------------------------- */
#define TLV_DIA_ON          				0x0001
#define TLV_DIA_OFF          				0x0000

static uint16_t usData;																				// ИСпользуется для SPI

static double const dPi = 3.14159265358979323846;// Иcпользуется для SPI

uint32_t ulSinArr[ SIN_STEPS ];

#define BUF_SIZE 200 // Size of package
#define AUDIO_FREQ 8000 // Sample rate

#define ECHO
//#define SIN

uint32_t bufferTxa[ BUF_SIZE ];
uint32_t bufferTxb[ BUF_SIZE ];

uint32_t  bufferRxa[ BUF_SIZE ];
uint32_t  bufferRxb[ BUF_SIZE ];


static void vTlv320ConfAnalogPath( uint16_t usConf );
static void vTlv320ConfDigitalPath( uint8_t ucConf );
static void vTlv320ConfPwd( uint8_t ucConf );
static void vTlv320ConfDigitalFormat( uint8_t ucConf );
static void vTlv320ConfSampleRate( uint8_t ucConf );
static void vTlv320DigitalInterfaceAct( uint8_t ucOn );
static void vTlv320Reset( void );
static void vTlv320MakeSin( void );

extern __builtin_quad audioTcbTxA;
extern __builtin_quad audioTcbRxA;
extern __builtin_quad audioTcbTxB;
extern __builtin_quad audioTcbRxB;


static void memcpy_stereo_to_mono(uint32_t *dst, uint32_t* src, uint32_t size)
{
	int i;
	for(i = 0; i < size; i++){
		dst[i] = (src[i*2]&0xFFFF) | (src[i*2+1]&0xFFFF) << 16;
	}
}

uint32_t FlagA = 1;
__attribute__((interrupt))
static void vTlv320_DmaIrqHandler( void )
{
	if (FlagA == 1)
	{
		if (HAL_DMA_GetDcAddr(5) < (uint32_t)bufferTxa || HAL_DMA_GetDcAddr(5) > (uint32_t)bufferTxa + BUF_SIZE){
			FlagA = 0;
			memcpy_stereo_to_mono(bufferTxa, bufferRxa, BUF_SIZE/2);
		}
	}
	else
	{
		if (HAL_DMA_GetDcAddr(5) < (uint32_t)bufferTxb || HAL_DMA_GetDcAddr(5) > (uint32_t)bufferTxb + BUF_SIZE){
			FlagA = 1;
			memcpy_stereo_to_mono(bufferTxb, bufferRxb, BUF_SIZE/2);
		}
	}
}

void vTlv320Init( uint32_t ulClkFreq )
{
	SPI_SOFT_type xSpiConf;
	TLV320_DIGFORMAT_type xDigitalFormatConf;
	AUDIO_I2S_type xI2Sconf;

	// Setup I2S Interface
	xI2Sconf.Role = ROLE_SLAVE; // Slave
	xI2Sconf.Mode = MODE_I2S;
	xI2Sconf.Standart = STANDART_PHILLIPS;
	xI2Sconf.TfsPos = 0;
	xI2Sconf.SwapLR = 0;
	xI2Sconf.DataLen = 16; // 16 bit audio
	xI2Sconf.AudioFreq = AUDIO_FREQ;

	#ifdef ECHO
	HAL_AUDIO_StartDMA_TX_2BUF( 5, LX_AUDIO0, bufferTxa, bufferTxb, BUF_SIZE>>1, 0 ); // BUF_SIZE>>1 because of mono sound
	HAL_AUDIO_StartDMA_RX_2BUF( 9, LX_AUDIO0, bufferRxa, bufferRxb, BUF_SIZE, vTlv320_DmaIrqHandler );
	#endif

	#ifdef SIN
	HAL_AUDIO_StartDMA_TX_2BUF( 5, LX_AUDIO0, ulSinArr, ulSinArr, SIN_STEPS, 0 ); // BUF_SIZE>>1 because of mono sound
	#endif

	HAL_AUDIO_I2SInit( LX_AUDIO0, &xI2Sconf );

	vTlv320MakeSin();


	// Settings for soft ssi
    xSpiConf.ulClkFreq = ulClkFreq;
	xSpiConf.ulSpiFreq = 50;
	xSpiConf.ulCPOL = 0;
	xSpiConf.ulCPHA = 0;
	xSpiConf.ulDataLen = 16;

	vSpiDrvInit( xSpiConf );

	// Setup I2S for TLV320
	xDigitalFormatConf.b.MS = 1;																	// TLV320 - master
	xDigitalFormatConf.b.LRSWAP = 0;																//
	xDigitalFormatConf.b.LRP = 0;																	//
	xDigitalFormatConf.b.IWL = DATA_16B;															// 16 bit audio
	xDigitalFormatConf.b.FOR = FORMAT_I2S;															// format of data - i2s

	// Setup registers of TLV320
	vTlv320DigitalInterfaceAct( TLV_DIA_OFF );
	vTlv320Reset();
	vTlv320ConfPwd( 0 );
	vTlv320ConfDigitalPath( 0 );
	vTlv320SetLineVolume( TLV_RLLIV_MUTED, CH_LR );
	vTlv320ConfAnalogPath( TLV_AAPC_DAC_SELECTED | TLV_AAPC_INSEL_MIC | TLV_AAPC_MIC_20DB );
	vTlv320SetHeadphoneVolume( TLV_RLCHV_MAX, CH_LR );
	vTlv320ConfDigitalFormat( xDigitalFormatConf.ucReg );
	vTlv320ConfSampleRate( SRC_USB_ADC8_DAC8 );
	vTlv320DigitalInterfaceAct( TLV_DIA_ON );

	HAL_DMA_WriteDC( 9, &audioTcbRxA );
	HAL_DMA_WriteDC( 5, &audioTcbTxA );
}

// Задание громкости линейного входа
void vTlv320SetLineVolume( uint8_t ucVolume, TLV320_CHANNEL_type eCh )
{
	uint16_t usVolume = ucVolume & 0x1F;
	usData = 0;

	if( usVolume == 0 )
		usVolume = 0x0080;																			// Muted

	switch( eCh )
	{
		case CH_L:
			usData = TLV_REG_LLIV << 9 | usVolume;
			vSpiDrvWriteData( &usData );
			break;
		case CH_R:
			usData = TLV_REG_RLIV << 9 | usVolume;
			vSpiDrvWriteData( &usData );
			break;
		case CH_LR:
			usData = TLV_REG_LLIV << 9 | TLV_RLLIV_LRS | usVolume ;
			vSpiDrvWriteData( &usData );
			break;
	}
}

// Задание громкости каналов наушников
void vTlv320SetHeadphoneVolume( uint8_t ucVolume, TLV320_CHANNEL_type eCh )
{
	usData = 0;
	ucVolume &= TLV_RLCHV_MAX;

	switch( eCh )
	{
		case CH_L:
			usData = TLV_REG_LCHV << 9 | ( ucVolume & 0xFF );
			vSpiDrvWriteData( &usData );
			break;
		case CH_R:
			usData = TLV_REG_RCHV << 9 | ( ucVolume & 0xFF );
			vSpiDrvWriteData( &usData );
			break;
		case CH_LR:
			usData = TLV_REG_LCHV << 9 | TLV_RLCHV_LRS | ( ucVolume & 0xFF );
			vSpiDrvWriteData( &usData );
			break;
	}
}

// Настройка аналоговой части
void vTlv320ConfAnalogPath( uint16_t usConf )
{
	usData = 0;
	usData = TLV_REG_AAPC << 9 | ( usConf & 0x1FF );
	vSpiDrvWriteData( &usData );
}

// Настройка цифровой части
void vTlv320ConfDigitalPath( uint8_t ucConf )
{
	usData = 0;
	usData = TLV_REG_DAPC << 9 | ( ucConf & 0x0F );
	vSpiDrvWriteData( &usData );

}

// Настройка питания
void vTlv320ConfPwd( uint8_t ucConf )
{
	usData = 0;
	usData = TLV_REG_PWDC << 9 | ( ucConf & 0xFF );
	vSpiDrvWriteData( &usData );
}

// Настройка формата аудиоданных
void vTlv320ConfDigitalFormat( uint8_t ucConf )
{
	usData = 0;
	usData = TLV_REG_DAIF << 9 | ( ucConf & 0x7F );
	vSpiDrvWriteData( &usData );

}

// Настройка частоты воборки
void vTlv320ConfSampleRate( uint8_t ucConf )
{
	usData = 0;
	usData = TLV_REG_SRC << 9 | ( ucConf & 0xFF );
	vSpiDrvWriteData( &usData );
}

// Включение цифрового интерфейса
void vTlv320DigitalInterfaceAct( uint8_t ucOn )
{
	if( ucOn > 1 )
		ucOn = 1;

	usData = 0;
	usData = TLV_REG_DIA << 9 | ( ucOn & 0x01 );
	vSpiDrvWriteData( &usData );
}

// Сброс устройства
void vTlv320Reset( void )
{
	usData = 0;
	usData = TLV_REG_RESET << 9;
	vSpiDrvWriteData( &usData );
}

void vTlv320MakeSin( void )
{
    uint16_t usI;
    int16_t sData = 0;
    double dK;

    dK = 2 * dPi / SIN_STEPS;

    for( usI = 0; usI < SIN_STEPS; usI++ )
    {
    	sData = SIN_MAX * sin( dK * usI );
//    	lSin[ usI ] = sData;
        ulSinArr[ usI ] = sData << 16 | ( sData & 0xFFFF );
    }
}
