/**
 ******************************************************************************
 * @file    main.c
 * @author  Milandr Application Team
 * @version V0.1.0
 * @date    17/12/2025
 * @brief   Main program body.
 ******************************************************************************
 * <br><br>
 *
 * THE PRESENT FIRMWARE IS FOR GUIDANCE ONLY. IT AIMS AT PROVIDING CUSTOMERS
 * WITH CODING INFORMATION REGARDING MILANDR'S PRODUCTS IN ORDER TO FACILITATE
 * THE USE AND SAVE TIME. MILANDR SHALL NOT BE HELD LIABLE FOR ANY
 * DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES RESULTING
 * FROM THE CONTENT OF SUCH FIRMWARE AND/OR A USE MADE BY CUSTOMERS OF THE
 * CODING INFORMATION CONTAINED HEREIN IN THEIR PRODUCTS.
 *
 * <h2><center>&copy; COPYRIGHT 2025 Milandr</center></h2>
 */

/* Includes ------------------------------------------------------------------*/
#include "MDR32FxQI_bkp.h"
#include "MDR32FxQI_eeprom.h"
#include "MDR32FxQI_port.h"
#include "MDR32FxQI_rst_clk.h"
#include "MDR32FxQI_uart.h"
#include "MDR32FxQI_utils.h"

/* Private define ------------------------------------------------------------*/
#define RESET_FLAG_OE

#define LED_PORT MDR_PORTC
#define LED_VD3  PORT_Pin_0
#define LED_VD4  PORT_Pin_1
#define LED_ALL  (LED_VD3 | LED_VD4)

/* Private macro -------------------------------------------------------------*/
#define GET_UART_TX_US(SPEED, BITS) ((((1000000000U / (SPEED)) * (BITS + 2)) + 999) / 1000) // (кол-во нс для передачи одного бита * кол-во бит в посылке (+START,STOP)) / 1000 = кол-во мкс для одной посылки

/* Private variables ---------------------------------------------------------*/
uint32_t count, tmp;

/* Private functions ---------------------------------------------------------*/
void CLK_Config(void);
void UART_Config(void);
void LED_Config(void);

void UART_SoftReset(MDR_UART_TypeDef* UARTx, uint32_t speed_uart, uint32_t word_length);
void UART_ResetFlagOE(MDR_UART_TypeDef* UARTx);
void UART_Error(void);

int main(void)
{
    CLK_Config();
    UART_Config();
    LED_Config();
    DELAY_Init(DELAY_MODE_PROGRAM);

    // Заполнение буфера FIFO_TX
    count = 1;
    while (UART_GetFlagStatus(MDR_UART2, UART_FLAG_TXFF) != SET) {
        UART_SendData(MDR_UART2, count++);
    }

    // Ожидание завершения отправки (буфер приёмника полон)
    while (UART_GetFlagStatus(MDR_UART2, UART_FLAG_BUSY) == SET) { }

    // Заполнение буфера FIFO_TX
    while (UART_GetFlagStatus(MDR_UART2, UART_FLAG_TXFF) != SET) {
        UART_SendData(MDR_UART2, count++);
    }

    // Программный сброс UART
    UART_SoftReset(MDR_UART2, 115200, 8);

    // Инициализация UART
    UART_Config();

    // Отправка данных
    for (count = 10; count < 26; count++) {
        UART_SendData(MDR_UART2, count);
        while (UART_GetFlagStatus(MDR_UART2, UART_FLAG_BUSY) == SET) { }
        tmp = UART_ReceiveData(MDR_UART2);
        if (tmp != count) {
            UART_Error();
        }
    }

    while (1) {
        PORT_SetBits(LED_PORT, LED_ALL); // LED ON
        DELAY_WaitMs(500);

        PORT_ResetBits(LED_PORT, LED_ALL); // LED OFF
        DELAY_WaitMs(500);
    }
}

void UART_Error(void)
{
    while (1) {
        PORT_SetBits(LED_PORT, LED_VD3); // LED ON
        DELAY_WaitMs(500);

        PORT_ResetBits(LED_PORT, LED_VD3); // LED OFF
        DELAY_WaitMs(500);
    }
}

void UART_SoftReset(MDR_UART_TypeDef* UARTx, uint32_t speed_uart, uint32_t word_length)
{
    UARTx->IMSC  = 0x00; // Запрещение прерываний
    UARTx->DMACR = 0x00; // Запрещение запросов DMA

    UARTx->CR &= ~(1 << 0); // Запрещение работы приёмопередатчика UART, при этом перед остановкой приёмопередатчик UART завершает приём/передачу текущего символа
    // Ожидание завершения последней транзакции. Бит BUSY не используется, т.к. в случае непустого FIFO передатчика BUSY = 1, даже если UARTEN = 0
    DELAY_WaitUs(GET_UART_TX_US(speed_uart, word_length));

    // При переполнении может остаться флаг OE (СП, п. "Бит переполнения буфера")
#ifdef RESET_FLAG_OE
    UART_ResetFlagOE(UARTx);
#else
    if (!(UARTx->LCR_H & UART_FIFO_ON)) {
        UARTx->LCR_H = UART_FIFO_ON; // Включить FIFO, чтобы сбросить буфер приёмника и передатчика
    }
#endif

    UARTx->IBRD  = 0x00;
    UARTx->FBRD  = 0x00;
    UARTx->LCR_H = 0x00; // Сброс FIFO передатчика и приёмника

    UARTx->RSR_ECR = 0x00;   // Сброс ошибок RSR_ECR
    UARTx->ICR     = 0xFFFF; // Сброс запросов прерываний ICR

    UARTx->ILPR = 0x00; // Установка оставшихся регистров значениями по сбросу
    UARTx->CR   = 0x0300;
    UARTx->IFLS = 0x12;
}

void UART_ResetFlagOE(MDR_UART_TypeDef* UARTx)
{
    volatile int temp = 0;

    MDR_PORTF->ANALOG &= ~((1 << 1) | (1 << 0)); // Отключение выводов UART RX и TX

    if (!(UARTx->LCR_H & UART_FIFO_ON)) {
        UARTx->LCR_H |= UART_FIFO_ON; // Включить FIFO, чтобы сбросить буфер приёмника и передатчика
    } else {
        UARTx->LCR_H &= ~UART_FIFO_ON; // Выключить FIFO, чтобы сбросить FIFO приёмника и передатчика
    }

    UARTx->CR |= (1 << 7) | (1 << 0); // Разрешение работы в режиме шлейфа

    UARTx->DR = 0x00;                      // Отправка, чтобы после приёма при чтении регистра DR бит OE установился, сообщая о том, что ранее было переполнение и потеря принимаемых символов
    while (UARTx->FR & UART_FLAG_BUSY) { } // Ожидание окончания приёма
    temp = UARTx->DR;

    UARTx->CR &= ~(1 << 0); // Запрещение работы приёмопередатчика UART

    MDR_PORTF->ANALOG |= ((1 << 1) | (1 << 0)); // Включение выводов RX и TX
}

void CLK_Config(void)
{
    RST_CLK_DeInit();

    RST_CLK_HSEconfig(RST_CLK_HSE_ON);
    while (RST_CLK_HSEstatus() != SUCCESS) { }

    RST_CLK_CPU_PLLconfig(RST_CLK_CPU_PLLsrcHSEdiv1, RST_CLK_CPU_PLLmul10);
    RST_CLK_CPU_PLLcmd(ENABLE);
    while (RST_CLK_CPU_PLLstatus() != SUCCESS) { }

    RST_CLK_CPU_PLLuse(ENABLE);
    RST_CLK_CPUclkPrescaler(RST_CLK_CPUclkDIV1);

    RST_CLK_PCLKcmd(RST_CLK_PCLK_EEPROM, ENABLE);
    EEPROM_SetLatency(EEPROM_Latency_3);
    RST_CLK_PCLKcmd(RST_CLK_PCLK_EEPROM, DISABLE);

    RST_CLK_PCLKcmd(RST_CLK_PCLK_BKP, ENABLE);
    BKP_DUccMode(BKP_DUcc_upto_80MHz);

    RST_CLK_CPUclkSelection(RST_CLK_CPUclkCPU_C3);

    SystemCoreClockUpdate();
}

void UART_Config(void)
{
    UART_InitTypeDef UART_InitStructure;
    PORT_InitTypeDef PortInit;

    RST_CLK_PCLKcmd(RST_CLK_PCLK_PORTF, ENABLE);

    PORT_StructInit(&PortInit);

    PortInit.PORT_Pin   = PORT_Pin_0 | PORT_Pin_1;
    PortInit.PORT_MODE  = PORT_MODE_DIGITAL;
    PortInit.PORT_SPEED = PORT_SPEED_FAST;
    PortInit.PORT_FUNC  = PORT_FUNC_OVERRID;

    PORT_Init(MDR_PORTF, &PortInit);

    RST_CLK_PCLKcmd(RST_CLK_PCLK_UART2, ENABLE);
    UART_BRGInit(MDR_UART2, UART_HCLKdiv1);

    UART_StructInit(&UART_InitStructure);
    UART_InitStructure.UART_BaudRate            = 115200;
    UART_InitStructure.UART_WordLength          = UART_WordLength8b;
    UART_InitStructure.UART_StopBits            = UART_StopBits1;
    UART_InitStructure.UART_Parity              = UART_Parity_No;
    UART_InitStructure.UART_FIFOMode            = UART_FIFO_ON;
    UART_InitStructure.UART_HardwareFlowControl = UART_HardwareFlowControl_RXE | UART_HardwareFlowControl_TXE;

    UART_Init(MDR_UART2, &UART_InitStructure);

    UART_ITConfig(MDR_UART2, UART_IT_TX | UART_IT_RX | UART_IT_RT | UART_IT_OE, ENABLE);
    UART_DMACmd(MDR_UART2, UART_DMA_RXE | UART_DMA_TXE | UART_DMA_ONERR, ENABLE);

    UART_Cmd(MDR_UART2, ENABLE);
}

void LED_Config(void)
{
    PORT_InitTypeDef GPIOInitStruct;

    RST_CLK_PCLKcmd(RST_CLK_PCLK_PORTC, ENABLE);
    PORT_StructInit(&GPIOInitStruct);

    GPIOInitStruct.PORT_Pin     = LED_ALL;
    GPIOInitStruct.PORT_OE      = PORT_OE_OUT;
    GPIOInitStruct.PORT_SPEED   = PORT_SPEED_SLOW;
    GPIOInitStruct.PORT_PULL_UP = PORT_PULL_UP_ON;
    GPIOInitStruct.PORT_MODE    = PORT_MODE_DIGITAL;
    PORT_Init(LED_PORT, &GPIOInitStruct);
}

/******************* (C) COPYRIGHT 2025 Milandr ********************************
 *
 * END OF FILE main.c */
