/**
 ******************************************************************************
 * @file    main.c
 * @author  Milandr Application Team
 * @version V1.0.1
 * @date    05/12/2025
 * @brief   This example shows how to use external int from port configuring in input mode and output to other port.
 ******************************************************************************
 * THE PRESENT FIRMWARE IS FOR GUIDANCE ONLY. IT AIMS AT PROVIDING CUSTOMERS
 * WITH CODING INFORMATION REGARDING MILANDR'S PRODUCTS IN ORDER TO FACILITATE
 * THE USE AND SAVE TIME. MILANDR SHALL NOT BE HELD LIABLE FOR ANY
 * DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES RESULTING
 * FROM THE CONTENT OF SUCH FIRMWARE AND/OR A USE MADE BY CUSTOMERS OF THE
 * CODING INFORMATION CONTAINED HEREIN IN THEIR PRODUCTS.
 */

/* Includes ------------------------------------------------------------------*/
#include "MDR1986VE8T.h"
#include "spec.h"

#include "mdr32f8_config.h"
#include "mdr32f8_port.h"
#include "mdr32f8_clkctrl.h"
#include "mdr32f8_ebc.h"

/* Private typedef -----------------------------------------------------------*/
PORT_InitTypeDef PORT_InitStructure;

/* Private define ------------------------------------------------------------*/
#define VD7  PORT_Pin_16
#define VD8  PORT_Pin_17
#define VD9  PORT_Pin_18
#define VD10 PORT_Pin_19
#define VD11 PORT_Pin_20
#define VD12 PORT_Pin_21
#define VD13 PORT_Pin_22
#define VD14 PORT_Pin_23

// #define SET_CUSTOM_EBC_WS

#define CUSTOM_EBC_WS_SETUP  1
#define CUSTOM_EBC_WS_ACTIVE 2
#define CUSTOM_EBC_WS_HOLD   1

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
void Blinky(uint32_t VD);

/* Private functions ---------------------------------------------------------*/
void Delay(__IO uint32_t nCount)
{
    for (; nCount != 0; nCount--) { }
}

/**
 * @brief  Main program.
 * @param  None
 * @retval None
 */
int main(void)
{
#ifdef SET_CUSTOM_EBC_WS
    // Key to access external bus control
    UNLOCK_UNIT(EXT_BUS_CNTR_key);
    EBC_RGNx_WSConfig(RGN0, CUSTOM_EBC_WS_SETUP, CUSTOM_EBC_WS_ACTIVE, CUSTOM_EBC_WS_HOLD);
#endif

    // Key to access clock control
    UNLOCK_UNIT(CLK_CNTR_key);
    // Key to access fault control
    UNLOCK_UNIT(FT_CNTR_key);
    // Key to access BKP control
    UNLOCK_UNIT(BKP_key);

    /* Set CLKCTRL to default */
    CLKCTRL_DeInit();

    /* Enable HSE0 clock */
    CLKCTRL_HSEconfig(CLKCTRL_HSE0_CLK_ON);

    /* Check HSE success, enable PLL0, check PLL0 success, select MAX_CLK src */
    while (CLKCTRL_HSEstatus(CLKCTRL_HSEn_STAT_HSE0_RDY) != SUCCESS) { }

    /* Select PLL0 clk src, PLL0_N, PLL0_Q to get FINT = FIN*(PLLn_N)/(PLLn_Q+1) */
    CLKCTRL_CPU_PLLconfig(PLL0, CLKCTRL_PLLn_CLK_SELECT_HSE0div1, PLL_DIVQ_Q_1_DV, 8); // PLLn, SRC, Q, DV, N
    while (CLKCTRL_CPU_PLLstatus(0) != SUCCESS) { }

    BKPCNTR_SRILOWconfig(BKPCNTR_SRILOW_upto_50MHz);
    CLKCTRL_MAX_CLKSelection(CLKCTRL_MAX_CLK_PLL0);

    /* Enable PORTx clk */
    CLKCTRL_PER0_CLKcmd(CLKCTRL_PER0_CLK_MDR_PORTE_EN, ENABLE);
    CLKCTRL_PER0_CLKcmd(CLKCTRL_PER0_CLK_MDR_PORTC_EN, ENABLE);

    /* Allow write to PORT regs */
    UNLOCK_UNIT(PORTC_key);
    UNLOCK_UNIT(PORTE_key);

    /* Configure PORTC pins [16:23] for output */
    PORT_InitStructure.PORT_Pin = (PORT_Pin_16 | PORT_Pin_17 | PORT_Pin_18 | PORT_Pin_19 |
                                   PORT_Pin_20 | PORT_Pin_21 | PORT_Pin_22 | PORT_Pin_23);

    PORT_InitStructure.PORT_SOE     = PORT_SOE_OUT;
    PORT_InitStructure.PORT_SANALOG = PORT_SANALOG_DIGITAL;
    PORT_InitStructure.PORT_SPD     = PORT_SPD_OFF;
    PORT_InitStructure.PORT_SPWR    = PORT_SPWR_10;

    PORT_Init(PORTC, &PORT_InitStructure);

    /* Configure PORTC pins [18,21,22] for input */
    PORT_InitStructure.PORT_Pin = (PORT_Pin_18 | PORT_Pin_21 | PORT_Pin_22);

    PORT_InitStructure.PORT_SOE     = PORT_SOE_IN;
    PORT_InitStructure.PORT_SANALOG = PORT_SANALOG_DIGITAL;
    PORT_InitStructure.PORT_SPD     = PORT_SPD_OFF;
    PORT_InitStructure.PORT_SPWR    = PORT_SPWR_10;
    /* IRQ enable and set active irq level */
    PORT_InitStructure.PORT_SIE = PORT_SIE_ON;
    PORT_InitStructure.PORT_SIT = PORT_SIT_LOW;

    PORT_Init(PORTE, &PORT_InitStructure);

    /* PORTE IRQ enable */
    NVIC_EnableIRQ(PORTE_IRQn);

    /*Main cycle*/
    while (1) { }
}

void Blinky(uint32_t VD)
{
    PORT_SetBits(PORTC, VD);
    Delay(5000);
    PORT_ResetBits(PORTC, VD);
    Delay(5000);
}

void IRQ_PORTE_Handler()
{
    Blinky(VD7 | VD8 | VD9 | VD10 | VD11 | VD12 | VD13 | VD14);
    PORTE->CIR = 0xFFFFFFFF;
}

//-----------------------------assert_param--------------------------------
#if (USE_ASSERT_INFO == 1)
void assert_failed(uint32_t file_id, uint32_t line)
{
    /* User can add his own implementation to report the source file ID and line number.
       Ex: printf("Wrong parameters value: file Id %d on line %d\r\n", file_id, line) */

    /* Infinite loop */
    while (1) {
    }
}
#elif (USE_ASSERT_INFO == 2)
void assert_failed(uint32_t file_id, uint32_t line, const uint8_t* expr)
{
    /* User can add his own implementation to report the source file ID, line number and
       expression text.
       Ex: printf("Wrong parameters value (%s): file Id %d on line %d\r\n", expr, file_id, line) */

    /* Infinite loop */
    while (1) {
    }
}
#endif /* USE_ASSERT_INFO */

/******************* (C) COPYRIGHT 2025 Milandr *********
 *
 * END OF FILE main.c */
