/**
 ******************************************************************************
 * @file    lcd.h
 * @author  Milandr Application Team
 * @version V0.1.0
 * @date    15/10/2025
 * @brief   This file contains all the functions prototypes for the LCD driver.
 ******************************************************************************
 * <br><br>
 *
 * THE PRESENT FIRMWARE IS FOR GUIDANCE ONLY. IT AIMS AT PROVIDING CUSTOMERS
 * WITH CODING INFORMATION REGARDING MILANDR'S PRODUCTS IN ORDER TO FACILITATE
 * THE USE AND SAVE TIME. MILANDR SHALL NOT BE HELD LIABLE FOR ANY
 * DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES RESULTING
 * FROM THE CONTENT OF SUCH FIRMWARE AND/OR A USE MADE BY CUSTOMERS OF THE
 * CODING INFORMATION CONTAINED HEREIN IN THEIR PRODUCTS.
 *
 * <h2><center>&copy; COPYRIGHT 2025 Milandr</center></h2>
 ******************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef LCD_H_
#define LCD_H_

#ifdef __cplusplus
extern "C" {
#endif

/* Includes ------------------------------------------------------------------*/
#include "MDR32VF0xI_config.h"
#include "MDR32VF0xI_rst_clk.h"
#include "MDR32VF0xI_port.h"

/** @addtogroup LCD_Driver LCD Driver
 * @{
 */

/** @addtogroup LCD LCD
 * @{
 */

/** @addtogroup LCD_Exported_Types LCD Exported Types
 * @{
 */

/**
 * @brief LCD crystal numbers.
 */
typedef enum {
    LCD_CRYSTAL_1 = 0,
    LCD_CRYSTAL_2 = 1,
    LCD_BOTH      = 2
} LCD_Crystal;

/**
 * @brief Bit and byte output methods.
 */
typedef enum {
    LCD_MET_OR      = 0,
    LCD_MET_XOR     = 1,
    LCD_MET_NOT_OR  = 2,
    LCD_MET_NOT_XOR = 3,
    LCD_MET_AND     = 4,
    NUM_LCD_METHOD
} LCD_Method;

/** @} */ /* End of the group LCD_Exported_Types */

/** @addtogroup LCD_Exported_Defines LCD Exported Defines
 * @{
 */

/**
 * @brief Number of LCD crystal.
 */
#define NUM_LCD_CRYSTALS LCD_BOTH

/**
 * @brief LCD status flags.
 */
#define LCD_BUSY_Pos      7
#define LCD_BUSY_Msk      (uint32_t)(1 << LCD_BUSY_Pos)

#define LCD_ONOFF_Pos     5
#define LCD_ONOFF_Msk     (uint32_t)(1 << LCD_ONOFF_Pos)

#define LCD_RESET_Pos     4
#define LCD_RESET_Msk     (uint32_t)(1 << LCD_RESET_Pos)

/**
 * @brief LCD screen resolution (in pixels).
 */
#define LCD_MAX_X 127
#define LCD_MAX_Y 63

/** @} */ /* End of the group LCD_Exported_Defines */

/** @addtogroup LCD_Exported_Macros LCD Exported Macros
 * @{
 */

/**
 * @brief Specific LCD control commands.
 */
#define LCD_ON             LCD_WriteCmd(0x3F)
#define LCD_OFF            LCD_WriteCmd(0x3E)
#define LCD_START_LINE(x)  LCD_WriteCmd(0xC0 | (uint8_t)(x))
#define LCD_SET_PAGE(x)    LCD_WriteCmd(0xB8 | (uint8_t)(x))
#define LCD_SET_ADDRESS(x) LCD_WriteCmd(0x40 | (uint8_t)(x))

/** @} */ /* End of the group LCD_Exported_Macros */

/** @addtogroup LCD_Exported_Functions LCD Exported Functions
 * @{
 */
void LCD_Reset(void);
void LCD_Clear(void);

void LCD_Init(void);

void        LCD_SetCrystal(LCD_Crystal Crystal);
LCD_Crystal LCD_GetCrystal(void);

void       LCD_SetMethod(LCD_Method Method);
LCD_Method LCD_GetMethod(void);

void    LCD_WriteCmd(uint8_t Cmd);
uint8_t LCD_ReadCmd(void);

void    LCD_WriteData(uint8_t Data);
uint8_t LCD_ReadData(void);

/** @} */ /* End of the group LCD_Exported_Functions */

/** @} */ /* End of the group LCD */

/** @} */ /* End of the group LCD_Driver */

#ifdef __cplusplus
} // extern "C" block end
#endif

#endif /* LCD_H_ */

/*********************** (C) COPYRIGHT 2025 Milandr ****************************
 *
 * END OF FILE lcd.h */
