/**
 ******************************************************************************
 * @file    lcd_text.h
 * @author  Milandr Application Team
 * @version V0.1.0
 * @date    15/10/2025
 * @brief   This file contains all the functions prototypes for the symbol
 *          and text output to LCD.
 ******************************************************************************
 * <br><br>
 *
 * THE PRESENT FIRMWARE IS FOR GUIDANCE ONLY. IT AIMS AT PROVIDING CUSTOMERS
 * WITH CODING INFORMATION REGARDING MILANDR'S PRODUCTS IN ORDER TO FACILITATE
 * THE USE AND SAVE TIME. MILANDR SHALL NOT BE HELD LIABLE FOR ANY
 * DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES RESULTING
 * FROM THE CONTENT OF SUCH FIRMWARE AND/OR A USE MADE BY CUSTOMERS OF THE
 * CODING INFORMATION CONTAINED HEREIN IN THEIR PRODUCTS.
 *
 * <h2><center>&copy; COPYRIGHT 2025 Milandr</center></h2>
 ******************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef LCD_TEXT_H
#define LCD_TEXT_H

#ifdef __cplusplus
extern "C" {
#endif

/* Includes ------------------------------------------------------------------*/
#include "lcd_font_defs.h"

/** @addtogroup LCD_Driver LCD Driver
 * @{
 */

/** @addtogroup LCD_Text LCD Text
 * @{
 */

/** @addtogroup LCD_Text_Exported_Types LCD Text Exported Types
 * @{
 */

/**
 * @brief LCD text styles.
 */
typedef enum {
    LCD_StyleSimple,
    LCD_StyleBlink,
    LCD_StyleFlipFlop,
    LCD_StyleVibratory
} LCD_TextStyle;

/** @} */ /* End of the group LCD_Text_Exported_Types */

/** @addtogroup LCD_Text_Exported_Defines LCD Text Exported Defines
 * @{
 */

/**
 * @brief LCD font size.
 */
#define LCD_DEFAULT_FONT_WIDTH  6
#define LCD_DEFAULT_FONT_HEIGHT 8

/**
 * @brief LCD lines.
 */
#define LCD_INTER_LINE     2
#define LCD_LINE_MESSAGE_1 LCD_DEFAULT_FONT_HEIGHT + 4
#define LCD_LINE_MESSAGE_2 LCD_LINE_MESSAGE_1 + LCD_DEFAULT_FONT_HEIGHT + LCD_INTER_LINE
#define LCD_LINE_MESSAGE_3 LCD_LINE_MESSAGE_2 + LCD_DEFAULT_FONT_HEIGHT + LCD_INTER_LINE
#define LCD_LINE_MESSAGE_4 LCD_LINE_MESSAGE_3 + LCD_DEFAULT_FONT_HEIGHT + LCD_INTER_LINE
#define LCD_LINE_MESSAGE_5 LCD_LINE_MESSAGE_4 + LCD_DEFAULT_FONT_HEIGHT + LCD_INTER_LINE

/** @} */ /* End of the group LCD_Text_Exported_Defines */

/** @addtogroup LCD_Text_Exported_Macros LCD Text Exported Macros
 * @{
 */

/**
 * @brief Macro for calculation of an address of symbol description structure in the
 *        symbol description table. Gets symbol code and font description address,
 *        returns symbol description address.
 */
#define LCD_GET_CHAR_DATA_ADDR(Symbol) \
    (CurrentFont)->pData + (Symbol) * (CurrentFont)->Width*((((CurrentFont)->Height % 8) != 0) ? (1 + (CurrentFont)->Height / 8) : ((CurrentFont)->Height / 8))

/** @} */ /* End of the group LCD_Text_Exported_Macros */

/** @addtogroup LCD_Text_Exported_Variables LCD Text Exported Variables
 * @{
 */
extern LCD_Font* CurrentFont; /* Selected font (used for symbol output). */

/** @} */ /* End of the group LCD_Text_Exported_Variables */

/** @addtogroup LCD_Text_Exported_Functions LCD Text Exported Functions
 * @{
 */

/** @addtogroup LCD_Text_Print_Raw_Data LCD Text Print Raw Data
 * @{
 */
void LCD_PutPixel(uint32_t X, uint32_t Y);                                                       /* Pixel to screen output. */
void LCD_PutByte(uint8_t X, uint8_t Y, uint8_t data);                                            /* Byte to screen output. */
void LCD_PutPicture(uint8_t X, uint8_t Y, uint8_t SizeX, uint8_t SizeY, const uint8_t* Picture); /* Picture to screen output. */

/** @} */ /* End of the group LCD_Text_Print_Raw_Data */

/** @addtogroup LCD_Text_Print_Data_Using_Current_Font LCD Text Print Data Using Current Font
 * @{
 */

void LCD_PutSymbol(uint8_t X, uint8_t Y, char Symbol);        /* Output symbols to screen using current font. */
void LCD_PutString(uint8_t X, uint8_t Y, const char* String); /* Output string to screen using current font. */

/** @} */ /* End of the group LCD_Text_Print_Data_Using_Current_Font */

/** @} */ /* End of the group LCD_Text_Exported_Functions */

/** @} */ /* End of the group LCD_Text */

/** @} */ /* End of the group LCD_Driver */

#ifdef __cplusplus
} // extern "C" block end
#endif

#endif /* LCD_TEXT_H */

/*********************** (C) COPYRIGHT 2025 Milandr ****************************
 *
 * END OF FILE lcd_text.h */
