/**
 ******************************************************************************
 * @file    lcd.c
 * @author  Milandr Application Team
 * @version V0.1.0
 * @date    15/10/2025
 * @brief   This file provides all the LCD driver functions.
 ******************************************************************************
 * <br><br>
 *
 * THE PRESENT FIRMWARE IS FOR GUIDANCE ONLY. IT AIMS AT PROVIDING CUSTOMERS
 * WITH CODING INFORMATION REGARDING MILANDR'S PRODUCTS IN ORDER TO FACILITATE
 * THE USE AND SAVE TIME. MILANDR SHALL NOT BE HELD LIABLE FOR ANY
 * DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES RESULTING
 * FROM THE CONTENT OF SUCH FIRMWARE AND/OR A USE MADE BY CUSTOMERS OF THE
 * CODING INFORMATION CONTAINED HEREIN IN THEIR PRODUCTS.
 *
 * <h2><center>&copy; COPYRIGHT 2025 Milandr</center></h2>
 ******************************************************************************
 */

/* Includes ------------------------------------------------------------------*/
#include "lcd.h"
#include "lcd_io_config.h"

/** @addtogroup LCD_Driver LCD Driver
 * @{
 */

/** @addtogroup LCD LCD
 * @{
 */

/** @addtogroup LCD_Private_Macros LCD Private Macros
 * @{
 */
#define LCD_WAIT_BUSY  LCD_WhileStatus(LCD_BUSY_Msk)
#define LCD_WAIT_RESET LCD_WhileStatus(LCD_RESET_Msk)
#define LCD_WAIT_ON    LCD_WhileStatus(LCD_ONOFF_Msk)

/** @} */ /* End of the group LCD_Private_Macros */

/** @addtogroup LCD_Private_Variables LCD Private Variables
 * @{
 */
static LCD_Crystal LCD_CurrentCrystal = LCD_CRYSTAL_1; /* Currently selected crystal.*/
static LCD_Method  LCD_CurrentMethod;                  /* Current display method. */

/** @} */ /* End of the group LCD_Private_Variables */

/** @addtogroup LCD_Private_Function_Prototypes LCD Private Function Prototypes
 * @{
 */
static void     LCD_Clock(void);
static uint32_t LCD_GetStatus(void);
static void     LCD_WhileStatus(uint32_t status);
static void     LCD_Pause(void);

/** @} */ /* End of the group LCD_Private_Function_Prototypes */

/** @addtogroup LCD_Private_Functions LCD Private Functions
 * @{
 */

/**
 * @brief   Generate the clock pulse.
 * @param   None.
 * @return  None.
 */
static void LCD_Clock(void)
{
    LCD_Pause();
    PORT_WriteBit(LCD_CLOCK_PORT, LCD_CLOCK_PIN, SET);
    LCD_Pause();
    PORT_WriteBit(LCD_CLOCK_PORT, LCD_CLOCK_PIN, RESET);
}

/**
 * @brief   Get the current LCD status.
 * @param   None.
 * @return  Return the current LCD status.
 */
static uint32_t LCD_GetStatus(void)
{
    uint32_t ret;

    ret = LCD_ReadCmd();
    return ret;
}

/**
 * @brief   Wait until the specific LCD status changes.
 * @param   Status: Specify the LCD status.
 *          This parameter can be one of the following values: LCD_BUSY, LCD_ONOFF, LCD_RESET.
 * @return  None.
 */
static void LCD_WhileStatus(uint32_t Status)
{
    uint32_t Stat;

    for (Stat = LCD_GetStatus(); (Stat & Status) != 0; Stat = LCD_GetStatus()) { }
}

/**
 * @brief   Delay for approx. 15 CPU clocks.
 * @param   None.
 * @return  None.
 */
static void LCD_Pause(void)
{
    volatile uint32_t i;

    for (i = 15; i > 0; i--) { }
}

/** @} */ /* End of the group LCD_Private_Functions */

/** @addtogroup LCD_Exported_Functions LCD Exported Functions
 * @{
 */

/**
 * @brief   Reset the LCD.
 * @param   None.
 * @return  None.
 */
void LCD_Reset(void)
{
    PORT_WriteBit(LCD_RESET_PORT, LCD_RESET_PIN, RESET);
    LCD_Pause();
    PORT_WriteBit(LCD_RESET_PORT, LCD_RESET_PIN, SET);
}

/**
 * @brief   Clear the LCD screen.
 * @param   None.
 * @return  None.
 */
void LCD_Clear(void)
{
    uint8_t Page, Addr, Crystal;

    for (Crystal = LCD_CRYSTAL_1; Crystal < NUM_LCD_CRYSTALS; Crystal++) {
        LCD_SetCrystal((LCD_Crystal)Crystal);
        LCD_WAIT_BUSY;
        LCD_SET_ADDRESS(0);
        for (Page = 0; Page < 8; Page++) {
            LCD_SET_PAGE(Page);
            for (Addr = 0; Addr < 64; Addr++) {
                LCD_WriteData(0x00);
            }
        }
    }
}

/**
 * @brief   Initialize all LCD crystals.
 * @param   None.
 * @return  None.
 */
void LCD_Init(void)
{
    uint32_t         Crystal;
    PORT_InitTypeDef LCD_Pins = {
        .PORT_Pin       = PORT_PIN_0,
        .PORT_Mode      = PORT_MODE_DIGITAL,
        .PORT_Direction = PORT_DIRECTION_INPUT_OUTPUT,
        .PORT_Function  = PORT_FUNCTION_PORT,
        .PORT_Power     = PORT_POWER_NOMINAL_UPTO_2mA,
        .PORT_PullUp    = PORT_PULL_UP_OFF,
        .PORT_PullDown  = PORT_PULL_DOWN_OFF,
    };

    RST_CLK_PCLKCmd(LCD_DATA_PORT_CLK | LCD_CRYSTAL_PORT_CLK | LCD_CMD_DATA_PORT_CLK | LCD_RD_WR_PORT_CLK | LCD_CLOCK_PORT_CLK | LCD_RESET_PORT_CLK, ENABLE);

    LCD_Pins.PORT_Pin = LCD_RESET_PIN;
    PORT_Init(LCD_RESET_PORT, &LCD_Pins);
    PORT_WriteBit(LCD_RESET_PORT, LCD_RESET_PIN, RESET);

    LCD_Pins.PORT_Pin = LCD_DATA_BUS_8_PINS;
    PORT_Init(LCD_DATA_PORT, &LCD_Pins);

    LCD_Pins.PORT_Pin = LCD_CRYSTAL_PINS;
    PORT_Init(LCD_CRYSTAL_PORT, &LCD_Pins);

    LCD_Pins.PORT_Pin = LCD_CMD_DATA_PIN;
    PORT_Init(LCD_CMD_DATA_PORT, &LCD_Pins);

    LCD_Pins.PORT_Pin = LCD_RD_WR_PIN;
    PORT_Init(LCD_RD_WR_PORT, &LCD_Pins);

    LCD_Pins.PORT_Pin = LCD_CLOCK_PIN;
    PORT_Init(LCD_CLOCK_PORT, &LCD_Pins);

    LCD_Reset();
    LCD_Pause();

    for (Crystal = LCD_CRYSTAL_1; Crystal < NUM_LCD_CRYSTALS; Crystal++) {
        LCD_SetCrystal((LCD_Crystal)Crystal);
        LCD_WAIT_BUSY;
        LCD_ON;
        LCD_WAIT_ON;
        LCD_START_LINE(0);
    }
}

/**
 * @brief   Set the current LCD crystal.
 * @param   Crystal: Specify the current LCD crystal.
 * @return  None.
 */
void LCD_SetCrystal(LCD_Crystal Crystal)
{
    PORT_WriteBit(LCD_CRYSTAL_PORT, LCD_CRYSTAL_PINS, RESET);
    PORT_WriteBit(LCD_CRYSTAL_PORT, ((Crystal + 1) << LCD_CRYSTAL_Pos), SET);
    LCD_Clock();
    LCD_CurrentCrystal = Crystal;
}

/**
 * @brief   Get the current LCD crystal.
 * @param   None.
 * @return  Return the current LCD crystal.
 */
LCD_Crystal LCD_GetCrystal(void)
{
    return LCD_CurrentCrystal;
}

/**
 * @brief   Set the current display method.
 * @param   Method: Specify the current display method.
 * @return  None.
 */
void LCD_SetMethod(LCD_Method Method)
{
    LCD_CurrentMethod = Method;
}

/**
 * @brief   Get the current display method.
 * @param   None.
 * @return  Return the current display method.
 */
LCD_Method LCD_GetMethod(void)
{
    return LCD_CurrentMethod;
}

/**
 * @brief   Write command to the current LCD crystal.
 * @param   Cmd: Specify the command to write.
 * @return  None.
 */
void LCD_WriteCmd(uint8_t Cmd)
{
    uint32_t PortData;

    /* Switch LCD to the command mode. */
    PORT_WriteBit(LCD_CMD_DATA_PORT, LCD_CMD_DATA_PIN, RESET);
    /* Switch LCD to the data input mode. */
    PORT_WriteBit(LCD_RD_WR_PORT, LCD_RD_WR_PIN, RESET);

    /* Switch LCD_DATA_PORT to the data output mode. */
    LCD_DATA_PORT->OE |= LCD_DATA_BUS_8_PINS;

    PortData = PORT_ReadInputData(LCD_DATA_PORT) & (~LCD_DATA_BUS_8_PINS);
    PortData |= Cmd << LCD_DATA_BUS_8_Pos;
    PORT_Write(LCD_DATA_PORT, PortData);

    LCD_Clock();

    /* Switch LCD_DATA_PORT to the data input mode. */
    LCD_DATA_PORT->OE &= ~LCD_DATA_BUS_8_PINS;

    /* Switch LCD to the data output mode. */
    PORT_WriteBit(LCD_RD_WR_PORT, LCD_RD_WR_PIN, SET);
}

/**
 * @brief   Read command from the current LCD crystal.
 * @param   None.
 * @return  Return command from the current LCD crystal.
 */
uint8_t LCD_ReadCmd(void)
{
    uint8_t Cmd;

    /* Switch LCD to the command mode. */
    PORT_WriteBit(LCD_CMD_DATA_PORT, LCD_CMD_DATA_PIN, RESET);
    LCD_Pause();
    PORT_WriteBit(LCD_CLOCK_PORT, LCD_CLOCK_PIN, SET);
    LCD_Pause();
    LCD_Pause();
    Cmd = (uint8_t)((PORT_ReadInputData(LCD_DATA_PORT) & LCD_DATA_BUS_8_PINS) >> LCD_DATA_BUS_8_Pos);
    PORT_WriteBit(LCD_CLOCK_PORT, LCD_CLOCK_PIN, RESET);

    return Cmd;
}

/**
 * @brief   Write data to the current LCD crystal.
 * @param   Data: Specify the data to write.
 * @return  None.
 */
void LCD_WriteData(uint8_t Data)
{
    uint32_t PortData;

    /* Switch LCD to the data mode. */
    PORT_WriteBit(LCD_CMD_DATA_PORT, LCD_CMD_DATA_PIN, SET);
    /* Switch LCD to the data input mode. */
    PORT_WriteBit(LCD_RD_WR_PORT, LCD_RD_WR_PIN, RESET);

    /* Switch LCD_DATA_PORT to the data output mode. */
    LCD_DATA_PORT->OE |= LCD_DATA_BUS_8_PINS;

    PortData = PORT_ReadInputData(LCD_DATA_PORT) & (~LCD_DATA_BUS_8_PINS);
    PortData |= Data << LCD_DATA_BUS_8_Pos;
    PORT_Write(LCD_DATA_PORT, PortData);

    LCD_Clock();

    /* Switch LCD_DATA_PORT to the data input mode. */
    LCD_DATA_PORT->OE &= ~LCD_DATA_BUS_8_PINS;

    /* Switch LCD to the data output mode. */
    PORT_WriteBit(LCD_RD_WR_PORT, LCD_RD_WR_PIN, SET);
}

/**
 * @brief   Read data from the current LCD crystal.
 * @param   None.
 * @return  Return data from the current LCD crystal.
 */
uint8_t LCD_ReadData()
{
    uint8_t Data;

    /* Switch LCD to the data mode. */
    PORT_WriteBit(LCD_CMD_DATA_PORT, LCD_CMD_DATA_PIN, SET);
    LCD_Pause();
    LCD_Clock(); /* Dummy reading is necessary to get correct data. */
    LCD_Pause();
    PORT_WriteBit(LCD_CLOCK_PORT, LCD_CLOCK_PIN, SET);
    LCD_Pause();
    LCD_Pause();
    Data = (uint8_t)((PORT_ReadInputData(LCD_DATA_PORT) & LCD_DATA_BUS_8_PINS) >> LCD_DATA_BUS_8_Pos);
    PORT_WriteBit(LCD_CLOCK_PORT, LCD_CLOCK_PIN, RESET);

    return Data;
}

/** @} */ /* End of the group LCD_Exported_Functions */

/*********************** (C) COPYRIGHT 2025 Milandr ****************************
 *
 * END OF FILE lcd.c */
