/**
 ******************************************************************************
 * @file    lcd_text.c
 * @author  Milandr Application Team
 * @version V0.1.0
 * @date    15/10/2025
 * @brief   This file provides all the functions for the symbol and text
 *          output to LCD.
 ******************************************************************************
 * <br><br>
 *
 * THE PRESENT FIRMWARE IS FOR GUIDANCE ONLY. IT AIMS AT PROVIDING CUSTOMERS
 * WITH CODING INFORMATION REGARDING MILANDR'S PRODUCTS IN ORDER TO FACILITATE
 * THE USE AND SAVE TIME. MILANDR SHALL NOT BE HELD LIABLE FOR ANY
 * DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES RESULTING
 * FROM THE CONTENT OF SUCH FIRMWARE AND/OR A USE MADE BY CUSTOMERS OF THE
 * CODING INFORMATION CONTAINED HEREIN IN THEIR PRODUCTS.
 *
 * <h2><center>&copy; COPYRIGHT 2025 Milandr</center></h2>
 ******************************************************************************
 */

/* Includes ------------------------------------------------------------------*/
#include <string.h>
#include "lcd.h"
#include "lcd_text.h"

/** @addtogroup LCD_Driver LCD Driver
 * @{
 */

/** @addtogroup LCD_Text LCD Text
 * @{
 */

/** @addtogroup LCD_Text_Exported_Variables LCD Text Exported Variables
 * @{
 */

LCD_Font* CurrentFont = &Font_ASCII_6x8; /* The selected font for text output. */

/** @} */ /* End of the group LCD_Text_Exported_Variables */

/** @addtogroup LCD_Text_Exported_Functions LCD Text Exported Functions
 * @{
 */

/** @addtogroup LCD_Text_Print_Raw_Data LCD Text Print Raw Data
 * @{
 */

/**
 * @brief  Output one pixel to the given screen coordinates.
 * @param  X: Specify the x-coordinate.
 * @param  Y: Specify the y-coordinate.
 * @return None.
 */
void LCD_PutPixel(uint32_t X, uint32_t Y)
{
    uint8_t Shift, Page, Data;

    if ((X > LCD_MAX_X) | (Y > LCD_MAX_Y)) {
        return;
    }

    LCD_SetCrystal((LCD_Crystal)(X / 64));
    X %= 64;

    Page  = (uint8_t)(Y / 8);
    Shift = (1 << (Y % 8));
    LCD_SET_PAGE(Page);
    LCD_SET_ADDRESS(X);
    Data = LCD_ReadData();
    switch (LCD_GetMethod()) {
        case LCD_MET_OR:
            Data = Data | Shift;
            break;
        case LCD_MET_XOR:
            Data = Data ^ Shift;
            break;
        case LCD_MET_NOT_OR:
            Data = ~Data | Shift;
            break;
        case LCD_MET_NOT_XOR:
            Data = ~Data ^ Shift;
            break;
        case LCD_MET_AND:
            Data = Data & ~Shift;
            break;
        default:
            return;
    }
    LCD_SET_ADDRESS(X);
    LCD_WriteData(Data);
}

/**
 * @brief  Output one byte to the given screen coordinates.
 * @param  X: Specify the x-coordinate.
 * @param  Y: Specify the y-coordinate.
 * @param  Data: Specify the byte to output.
 * @return None.
 */
void LCD_PutByte(uint8_t X, uint8_t Y, uint8_t Data)
{
    uint8_t TmpData, Page, ShiftNum, ShiftNumEx;

    if ((X > LCD_MAX_X) | (Y > LCD_MAX_Y)) {
        return;
    }

    /* Select crystal and X offset. */
    LCD_SetCrystal((LCD_Crystal)(X / 64));
    X %= 64;
    /* Determine first page and y offset. */
    Page       = Y / 8;
    ShiftNum   = Y % 8;
    ShiftNumEx = 8 - ShiftNum;

    /* First page. */
    /* Read current value. */
    LCD_SET_PAGE(Page);
    LCD_SET_ADDRESS(X);
    TmpData = LCD_ReadData();
    /* Write modified value. */
    LCD_SET_PAGE(Page);
    LCD_SET_ADDRESS(X);
    switch (LCD_GetMethod()) {
        case LCD_MET_OR:
            LCD_WriteData(TmpData | (Data << ShiftNum));
            break;
        case LCD_MET_XOR:
            LCD_WriteData(TmpData ^ (Data << ShiftNum));
            break;
        case LCD_MET_NOT_OR:
            LCD_WriteData(TmpData | ((Data ^ 0xFF) << ShiftNum));
            break;
        case LCD_MET_NOT_XOR:
            LCD_WriteData(TmpData ^ ((Data ^ 0xFF) << ShiftNum));
            break;
        case LCD_MET_AND:
            LCD_WriteData((TmpData & (0xFF >> ShiftNumEx)) | (Data << ShiftNum));
            break;
        default:
            return;
    }

    /* Second page (if any). */
    if (ShiftNum > 0) {
        /* Read current value. */
        LCD_SET_PAGE(Page + 1);
        LCD_SET_ADDRESS(X);
        TmpData = LCD_ReadData();
        /* Write modified value. */
        LCD_SET_PAGE(Page + 1);
        LCD_SET_ADDRESS(X);
        switch (LCD_GetMethod()) {
            case LCD_MET_OR:
                LCD_WriteData(TmpData | (Data >> ShiftNumEx));
                break;
            case LCD_MET_XOR:
                LCD_WriteData(TmpData ^ (Data >> ShiftNumEx));
                break;
            case LCD_MET_NOT_OR:
                LCD_WriteData(TmpData | ((Data ^ 0xFF) >> ShiftNumEx));
                break;
            case LCD_MET_NOT_XOR:
                LCD_WriteData(TmpData ^ ((Data ^ 0xFF) >> ShiftNumEx));
                break;
            case LCD_MET_AND:
                LCD_WriteData((TmpData & (0xFF << ShiftNum)) | (Data >> ShiftNumEx));
                break;
            default:
                return;
        }
    }
}

/**
 * @brief  Print the picture to the given screen coordinates.
 * @param  X: Specify the x-coordinate.
 * @param  Y: Specify the y-coordinate.
 * @param  SizeX: Specify the number of columns.
 * @param  SizeY: Specify the number of rows.
 * @param  Picture: The pointer to picture.
 * @return None.
 */
void LCD_PutPicture(uint8_t X, uint8_t Y, uint8_t SizeX, uint8_t SizeY, const uint8_t* Picture)
{
    uint8_t Cols, Rows, Line;

    Line = SizeY / 8;
    if (CurrentFont->Height % 8) {
        Line++;
    }

    for (Rows = 0; Rows < Line; Rows++) {
        for (Cols = 0; Cols < SizeX; Cols++) {
            LCD_PutByte(X + Cols, Y + (uint8_t)(Rows * 8), Picture[Cols + SizeX * Rows]);
        }
    }
}

/** @} */ /* End of the group LCD_Text_Print_Raw_Data */

/** @addtogroup LCD_Text_Print_Data_Using_Current_Font LCD Text Print Data Using Current Font
 * @{
 */

/**
 * @brief  Print one symbol to the given screen coordinates.
 * @param  X: Specify the x-coordinate.
 * @param  Y: Specify the y-coordinate.
 * @param  Symbol: Specify the symbol to print.
 * @return None.
 */
void LCD_PutSymbol(uint8_t X, uint8_t Y, char Symbol)
{
    uint8_t        Cols, Rows, Line;
    const uint8_t* Sym;

    Sym  = LCD_GET_CHAR_DATA_ADDR((uint32_t)Symbol);
    Line = (uint8_t)(CurrentFont->Height / 8);
    if (CurrentFont->Height % 8) {
        Line++;
    }

    for (Rows = 0; Rows < Line; Rows++) {
        for (Cols = 0; Cols < CurrentFont->Width; Cols++) {
            LCD_PutByte(X + Cols, Y + (uint8_t)(Rows * 8), Sym[Cols + CurrentFont->Width * Rows]);
        }
    }
}

/**
 * @brief  Print the string to the given screen coordinates.
 * @param  X: Specify the x-coordinate.
 * @param  Y: Specify the y-coordinate.
 * @param  String: The pointer to string.
 * @return None.
 */
void LCD_PutString(uint8_t X, uint8_t Y, const char* String)
{
    uint8_t Cols;

    for (Cols = 0; String[Cols]; Cols++) {
        LCD_PutSymbol(X + (uint8_t)(Cols * CurrentFont->Width), Y, String[Cols]);
    }
}

/** @} */ /* End of the group LCD_Text_Print_Data_Using_Current_Font */

/** @} */ /* End of the group LCD_Text_Exported_Functions */

/** @} */ /* End of the group LCD_Text */

/** @} */ /* End of the group LCD_Driver */

/*********************** (C) COPYRIGHT 2025 Milandr ****************************
 *
 * END OF FILE lcd_text.c */
