/**
 ******************************************************************************
 * @file    main.c
 * @author  Milandr Application Team
 * @version V0.1.0
 * @date    21/10/2025
 * @brief   Main program body.
 ******************************************************************************
 * THE PRESENT FIRMWARE IS FOR GUIDANCE ONLY. IT AIMS AT PROVIDING CUSTOMERS
 * WITH CODING INFORMATION REGARDING MILANDR'S PRODUCTS IN ORDER TO FACILITATE
 * THE USE AND SAVE TIME. MILANDR SHALL NOT BE HELD LIABLE FOR ANY
 * DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES RESULTING
 * FROM THE CONTENT OF SUCH FIRMWARE AND/OR A USE MADE BY CUSTOMERS OF THE
 * CODING INFORMATION CONTAINED HEREIN IN THEIR PRODUCTS.
 */

/* Includes ------------------------------------------------------------------*/
#include "MDR32VF0xI_bkp.h"
#include "MDR32VF0xI_flash.h"
#include "MDR32VF0xI_port.h"
#include "MDR32VF0xI_rst_clk.h"
#include "system_MDR32VF0xI_it.h"

#include "lcd.h"
#include "lcd_mdr_logo.h"
#include "lcd_text.h"

#include "stdio_uart.h"
#include <stdio.h>

/* Private define ------------------------------------------------------------*/
#define LED_PORT            MDR_PORTD
#define LED_PORT_CLK        RST_CLK_PCLK_PORTD
#define LED_PIN_1           PORT_PIN_5
#define LED_PIN_2           PORT_PIN_6
#define LED_PIN_3           PORT_PIN_7
#define LED_PIN_4           PORT_PIN_8
#define LED_PINS            (LED_PIN_1 | LED_PIN_2 | LED_PIN_3 | LED_PIN_4)

#define BTN_PORT            MDR_PORTA
#define BTN_PORT_CLK        RST_CLK_PCLK_PORTA
#define BTN_PIN_1           PORT_PIN_0
#define BTN_PIN_2           PORT_PIN_1
#define BTN_PIN_3           PORT_PIN_2
#define BTN_PIN_4           PORT_PIN_3
#define BTN_PINS            (BTN_PIN_1 | BTN_PIN_2 | BTN_PIN_3 | BTN_PIN_4)

#define MTIMER_PERIOD_TICKS 3276800

/* Private variables ---------------------------------------------------------*/
uint32_t         DelayCnt   = 0;
volatile uint8_t MTimerTick = 0;

/* Private function prototypes -----------------------------------------------*/
void ClockConfig(void);
void LedConfig(void);
void ButtonConfig(void);
void MachineTimerConfig(void);

/* Private functions ---------------------------------------------------------*/

/**
 * @brief  Machine timer interrupt handler.
 * @note   __INTERRUPT_MACHINE attribute is used because interrupt is configured in vector mode.
 * @param  None.
 * @return None.
 */
__INTERRUPT_MACHINE void MTIP_IRQHandler(void)
{
    CLINT_MTIMER_SetCompareTime(CLINT_MTIMER_GetTime() + MTIMER_PERIOD_TICKS);
    MTimerTick ^= 1;
}

/**
 * @brief  Main program.
 * @param  None.
 * @return None.
 */
int main()
{
    ClockConfig();
    LedConfig();
    ButtonConfig();
    MachineTimerConfig();

    STDIO_UART_Init();

    LCD_Init();
    LCD_SetMethod(LCD_MET_AND);
    LCD_Clear();
    LCD_PutPicture(8, 0, LCD_MDR_LOGO_SIZE_X, LCD_MDR_LOGO_SIZE_Y, MilandrLogo);
    LCD_PutString(54, 38, "MDR1206FI");
    LCD_PutSymbol(24, 40, 0x04); // *

    printf("MDR1206FI Init end!\r\n");

    while (1) {
        /* Button DOWN. */
        if (PORT_ReadInputDataBit(BTN_PORT, BTN_PIN_1) == SET) {
            LCD_PutSymbol(24, 48, 0);
        } else {
            LCD_PutSymbol(24, 48, 0x19);
        }

        /* Button LEFT. */
        if (PORT_ReadInputDataBit(BTN_PORT, BTN_PIN_2) == SET) {
            LCD_PutSymbol(16, 40, 0);
        } else {
            LCD_PutSymbol(16, 40, 0x1B);
        }

        /* Button RIGHT. */
        if (PORT_ReadInputDataBit(BTN_PORT, BTN_PIN_3) == SET) {
            LCD_PutSymbol(32, 40, 0);
        } else {
            LCD_PutSymbol(32, 40, 0x1A);
        }

        /* Button UP. */
        if (PORT_ReadInputDataBit(BTN_PORT, BTN_PIN_4) == SET) {
            LCD_PutSymbol(24, 33, 0);
        } else {
            LCD_PutSymbol(24, 33, 0x18);
        }

        /* WAKEUP1-3. */
        if (BKP_GetWAKEUPStatus(BKP_WAKEUP_1) == RESET) {
            LCD_PutString(12, 56, "WAKEUP_1");
        } else if (BKP_GetWAKEUPStatus(BKP_WAKEUP_2) == RESET) {
            LCD_PutString(12, 56, "WAKEUP_2");
        } else if (BKP_GetWAKEUPStatus(BKP_WAKEUP_3) == RESET) {
            LCD_PutString(12, 56, "WAKEUP_3");
        } else {
            LCD_PutString(12, 56, "        ");
        }

        /* Machine timer tick. */
        if (MTimerTick) {
            LCD_PutSymbol(80, 52, 0x5C);
            PORT_WriteBit(LED_PORT, LED_PIN_4, RESET);
        } else {
            LCD_PutSymbol(80, 52, 0x2F);
            PORT_WriteBit(LED_PORT, LED_PIN_4, SET);
        }

        /* Program delay. */
        if (DelayCnt++ >= 0x00000100) {
            DelayCnt = 0;
            PORT_ToggleBit(LED_PORT, LED_PIN_1);
        }
    }
}

/**
 * @brief  Clock configuration.
 * @param  None.
 * @return None.
 */
void ClockConfig(void)
{
    ErrorStatus ClockStatus;

    static const RST_CLK_HCLK_InitTypeDef CoreClock = {
        .RST_CLK_CPU_C1_Source      = RST_CLK_CPU_C1_CLK_SRC_HSE,
        .RST_CLK_PLLCPU_ClockSource = RST_CLK_PLLCPU_CLK_SRC_CPU_C1,
        .RST_CLK_PLLCPU_Multiplier  = RST_CLK_PLLCPU_MUL_6,
        .RST_CLK_CPU_C2_ClockSource = RST_CLK_CPU_C2_CLK_SRC_PLLCPU,
        .RST_CLK_CPU_C3_Prescaler   = RST_CLK_CPU_C3_PRESCALER_DIV_1,
        .RST_CLK_HCLK_ClockSource   = RST_CLK_CPU_HCLK_CLK_SRC_CPU_C3
    };

    RST_CLK_DeInit();
    RST_CLK_HCLK_ClkSelection(RST_CLK_CPU_HCLK_CLK_SRC_HSI);
    SystemCoreClockUpdate();

    RST_CLK_PCLKCmd(RST_CLK_PCLK_FLASH, ENABLE);

    RST_CLK_HSE_Cmd(ENABLE);
    ClockStatus = RST_CLK_HSE_GetStatus();
    if (ClockStatus == SUCCESS) {
        FLASH_SetLatency(FLASH_LATENCY_CYCLE_1);
        ClockStatus = RST_CLK_HCLK_Init(&CoreClock);
        if (ClockStatus == SUCCESS) {
            RST_CLK_PER1_C2_ClkSelection(RST_CLK_PER1_C2_CLK_SRC_PLLCPU);
        }
    }

    if (ClockStatus == ERROR) {
        FLASH_SetLatency(FLASH_LATENCY_CYCLE_0);
        RST_CLK_PER1_C2_ClkSelection(RST_CLK_PER1_C2_CLK_SRC_CPU_C1);
    }

    RST_CLK_PCLKCmd(RST_CLK_PCLK_FLASH, DISABLE);
    RST_CLK_PCLKCmd(RST_CLK_PCLK_BKP, ENABLE);
}

/**
 * @brief  LEDs configuration.
 * @param  None.
 * @return None.
 */
void LedConfig(void)
{
    static const PORT_InitTypeDef LedPins = {
        .PORT_Pin       = LED_PINS,
        .PORT_Mode      = PORT_MODE_DIGITAL,
        .PORT_Direction = PORT_DIRECTION_INPUT_OUTPUT,
        .PORT_Function  = PORT_FUNCTION_PORT,
        .PORT_Power     = PORT_POWER_NOMINAL_UPTO_2mA,
        .PORT_PullUp    = PORT_PULL_UP_OFF,
        .PORT_PullDown  = PORT_PULL_DOWN_OFF,
    };

    RST_CLK_PCLKCmd(LED_PORT_CLK, ENABLE);
    PORT_SetBits(LED_PORT, LED_PINS);
    PORT_Init(LED_PORT, &LedPins);
}

/**
 * @brief  Buttons configuration.
 * @param  None.
 * @return None.
 */
void ButtonConfig(void)
{
    static const PORT_InitTypeDef ButtonPins = {
        .PORT_Pin       = BTN_PINS,
        .PORT_Mode      = PORT_MODE_DIGITAL,
        .PORT_Direction = PORT_DIRECTION_INPUT,
        .PORT_Function  = PORT_FUNCTION_PORT,
        .PORT_Power     = PORT_POWER_NOMINAL_UPTO_2mA,
        .PORT_PullUp    = PORT_PULL_UP_OFF,
        .PORT_PullDown  = PORT_PULL_DOWN_OFF,
    };

    RST_CLK_PCLKCmd(BTN_PORT_CLK, ENABLE);
    PORT_Init(BTN_PORT, &ButtonPins);
}

/**
 * @brief  Machine timer configuration.
 * @param  None.
 * @return None.
 */
void MachineTimerConfig(void)
{
    RST_CLK_SystemTimer_SetPrescaler(2);
    CLINT_MTIMER_SetCompareTime(CLINT_MTIMER_GetTime() + MTIMER_PERIOD_TICKS);

    CLIC_EnableIRQ(MTIP_IRQn);
}

/**
 * @brief  Report the source file name, the source line number and expression
 *         text (if USE_ASSERT_INFO == 2) where the assert_param error has occurred.
 * @param  file: Pointer to the source file name.
 * @param  line: assert_param error line source number.
 * @param  expr: Reported expression text.
 * @return None.
 */
#if (USE_ASSERT_INFO == 1)
void assert_failed(uint8_t* file, uint32_t line)
{
    (void)file;
    (void)line;
    while (1) { }
}
#elif (USE_ASSERT_INFO == 2)
void assert_failed(uint8_t* file, uint32_t line, const uint8_t* expr)
{
    (void)file;
    (void)line;
    (void)expr;
    while (1) { }
}
#endif /* USE_ASSERT_INFO */

/*********************** (C) COPYRIGHT 2025 Milandr ****************************
 *
 * END OF FILE main.c */
