/**
 ******************************************************************************
 * @file    syscalls.c
 * @author  Milandr Application Team
 * @version V0.1.1
 * @date    15/10/2025
 * @brief   Implementation of newlib syscalls.
 ******************************************************************************
 * <br><br>
 *
 * THE PRESENT FIRMWARE IS FOR GUIDANCE ONLY. IT AIMS AT PROVIDING CUSTOMERS
 * WITH CODING INFORMATION REGARDING MILANDR'S PRODUCTS IN ORDER TO FACILITATE
 * THE USE AND SAVE TIME. MILANDR SHALL NOT BE HELD LIABLE FOR ANY
 * DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES RESULTING
 * FROM THE CONTENT OF SUCH FIRMWARE AND/OR A USE MADE BY CUSTOMERS OF THE
 * CODING INFORMATION CONTAINED HEREIN IN THEIR PRODUCTS.
 *
 * <h2><center>&copy; COPYRIGHT 2025 Milandr</center></h2>
 ******************************************************************************
 */

/* Includes ------------------------------------------------------------------*/
#include <sys/stat.h>
#include <sys/times.h>
#include <errno.h>
#include <unistd.h>

#include <MDR32VF0xI_uart.h>
#include "stdio_uart.h"

/** @addtogroup Syscalls Syscalls
 * @{
 */

/** @addtogroup Syscalls_Private_Defines SYSCALLS Private Defines
 * @{
 */
#define UNUSED(X) (void)(X)

/** @} */ /* End of the group Syscalls_Private_Defines */

/** @addtogroup Syscalls_Private_Variables Syscalls Private Variables
 * @{
 */
char*  __env[1] = { 0 };
char** environ  = __env;

/** @} */ /* End of the group Syscalls_Private_Variables */

/** @addtogroup Syscalls_Private_Function_Prototypes Syscalls Private Function Prototypes
 * @{
 */
void    _exit(int status);
int     _close(int file);
int     _execve(char* name, char** argv, char** env);
int     _fork(void);
int     _fstat(int file, struct stat* st);
int     _getpid(void);
int     _isatty(int file);
int     _kill(int pid, int sig);
int     _link(char* old, char* new);
int     _lseek(int file, int ptr, int dir);
int     _open(const char* name, int flags, int mode);
int     _read(int file, char* ptr, int len);
caddr_t _sbrk(int incr);
int     _stat(char* file, struct stat* st);
int     _times(struct tms* buf);
int     _unlink(char* name);
int     _wait(int* status);
int     _write(int file, char* ptr, int len);

/** @} */ /* End of the group Syscalls_Private_Function_Prototypes */

/** @addtogroup Syscalls_Private_Functions Syscalls Private Functions
 * @{
 */
void _exit(int status)
{
    _kill(status, -1);
    while (1) { }
}

int _close(int file)
{
    UNUSED(file);
    return -1;
}

int _execve(char* name, char** argv, char** env)
{
    UNUSED(name);
    UNUSED(argv);
    UNUSED(env);
    errno = ENOMEM;
    return -1;
}

int _fork(void)
{
    errno = EAGAIN;
    return -1;
}

int _fstat(int file, struct stat* st)
{
    UNUSED(file);
    st->st_mode = S_IFCHR;
    return 0;
}

int _getpid(void)
{
    return 1;
}

int _isatty(int file)
{
    UNUSED(file);
    return 1;
}

int _kill(int pid, int sig)
{
    UNUSED(pid);
    UNUSED(sig);
    errno = EINVAL;
    return -1;
}

int _link(char* old, char* new)
{
    UNUSED(old);
    UNUSED(new);
    errno = EMLINK;
    return -1;
}

int _lseek(int file, int ptr, int dir)
{
    UNUSED(file);
    UNUSED(ptr);
    UNUSED(dir);
    return 0;
}

int _open(const char* name, int flags, int mode)
{
    UNUSED(name);
    UNUSED(flags);
    UNUSED(mode);
    return -1;
}

int _read(int file, char* ptr, int len)
{
    UNUSED(file);
    int read = 0;

    if (!(file == STDIN_FILENO)) {
        errno = EBADF;
        return -1;
    }

    for (; len != 0; --len) {
        while (UART_GetFlagStatus(STDIO_UART, UART_FLAG_RXFE) == SET) { }
        *ptr++ = (char)UART_DATA(UART_ReceiveData(STDIO_UART));
        ++read;
    }

    return read;
}

caddr_t _sbrk(int incr)
{
    extern char  __heap_top[];   // End of the bss section (start of heap).
    extern char  __heap_limit[]; // End of the heap section (start of stack).
    char*        prev_heap_end;
    static char* heap_end = __heap_top;

    if (heap_end + incr > __heap_limit) {
        errno = ENOMEM;
        return (caddr_t)-1;
    }

    prev_heap_end = heap_end;
    heap_end += incr;

    return (caddr_t)prev_heap_end;
}

int _stat(char* file, struct stat* st)
{
    UNUSED(file);
    st->st_mode = S_IFCHR;
    return 0;
}

int _times(struct tms* buf)
{
    UNUSED(buf);
    return -1;
}

int _unlink(char* name)
{
    UNUSED(name);
    errno = ENOENT;
    return -1;
}

int _wait(int* status)
{
    UNUSED(status);
    errno = ECHILD;
    return -1;
}

int _write(int file, char* ptr, int len)
{
    UNUSED(file);
    int write = 0;

    if (!(file == STDOUT_FILENO || file == STDERR_FILENO)) {
        errno = EBADF;
        return -1;
    }

    for (; len != 0; --len) {
        while (UART_GetFlagStatus(STDIO_UART, UART_FLAG_TXFF) == SET) { }
        UART_SendData(STDIO_UART, (uint8_t)*ptr++);
        ++write;
    }

    return write;
}

/** @} */ /* End of the group Syscalls_Private_Functions */

/** @} */ /* End of the group Syscalls */

/*********************** (C) COPYRIGHT 2025 Milandr ****************************
 *
 * END OF FILE syscalls.c */
