/**
  ******************************************************************************
  * @file    main.c
  * @author  Milandr Application Team
  * @version V2.1.0
  * @date    22/06/2023
  * @brief   Main program body.
  ******************************************************************************
  * <br><br>
  *
  * THE PRESENT FIRMWARE IS FOR GUIDANCE ONLY. IT AIMS AT PROVIDING CUSTOMERS
  * WITH CODING INFORMATION REGARDING MILANDR'S PRODUCTS IN ORDER TO FACILITATE
  * THE USE AND SAVE TIME. MILANDR SHALL NOT BE HELD LIABLE FOR ANY
  * DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES RESULTING
  * FROM THE CONTENT OF SUCH FIRMWARE AND/OR A USE MADE BY CUSTOMERS OF THE
  * CODING INFORMATION CONTAINED HEREIN IN THEIR PRODUCTS.
  *
  * <h2><center>&copy; COPYRIGHT 2024 Milandr</center></h2>
  */

/* Includes ------------------------------------------------------------------*/

#include "MDR32F9Qx_eeprom.h"
#include "MDR32F9Qx_port.h"
#include "MDR32F9Qx_rst_clk.h"

/** @addtogroup __MDR1986BE4_StdPeriph_Examples MDR1986BE4 StdPeriph Examples
  * @{
  */

/** @addtogroup __MDR1986BE4_EVAL MDR1986BE4 Evaluation Board
  * @{
  */

/** @addtogroup EEPROM_Sector_Operations EEPROM_Sector_Operations
  * @{
  */

/* Private define ------------------------------------------------------------*/
#define EEPROM_PAGE_SIZE             (512)
#define MAIN_EEPAGE                  64
#define LED0_PIN                     PORT_Pin_11
#define LED0_PORT                    MDR_PORTA

/* Private macro -------------------------------------------------------------*/

/* Private variables ---------------------------------------------------------*/
static PORT_InitTypeDef PORT_InitStructure;
uint8_t EraseErrMM;
uint8_t WriteErrMM;
uint8_t EraseErrIM;
uint8_t WriteErrIM;

/* Private functions ---------------------------------------------------------*/

/**
  * @brief  Generation random values
  * @param  addr - memory address
  * @retval None
  */
uint32_t Pseudo_Rand(uint32_t addr)
{
    uint32_t hash = 0;
    uint32_t i = 0;
    uint8_t* key = (uint8_t *)&addr;

    for(i = 0; i < 4; i++)
    {
        hash += key[i];
        hash += (hash << 10);
        hash ^= (hash >> 6);
    }

    for(i = 0; i < 256; i++)
    {
        hash += (hash << 10);
        hash ^= (hash >> 6);
    }

    hash += (hash << 3);
    hash ^= (hash >> 11);
    hash += (hash << 15);
    return hash;
}

/**
  * @brief  Executes delay loop.
  * @param  nCount: specifies the delay.
  * @retval None
  */
void Delay(__IO uint32_t nCount)
{
    for(; nCount != 0; nCount--);
}

/**
  * @brief  Blink with LED1
  * @param  num - blinks number
  * @param  del - delay
  * @retval None
  */
void BlinkLED0(uint32_t num, uint32_t del)
{
    uint32_t cnt;

    for(cnt = 0; cnt < num; cnt++)
    {
        PORT_SetBits(LED0_PORT, LED0_PIN);
        Delay(del);
        PORT_ResetBits(LED0_PORT, LED0_PIN);
        Delay(del);
    }
}


/**
  * @brief  Main program.
  * @param  None
  * @retval None
  */
int main(void)
{
    uint32_t Address = 0;
    uint32_t Data = 0;
    uint32_t i = 0;
    EEPROM_Mem_Bank BankSelector = EEPROM_Main_Bank_Select;

    RST_CLK_DeInit();

    SystemCoreClockUpdate();

    /* Enables the clock on PORTA and EEPROM */
    RST_CLK_PCLKcmd((RST_CLK_PCLK_PORTA | RST_CLK_PCLK_EEPROM), ENABLE);

    /* Configure PORTA pins 11 for output to switch LED on/off */
    PORT_StructInit(&PORT_InitStructure);
    PORT_InitStructure.PORT_Pin   = LED0_PIN;
    PORT_InitStructure.PORT_OE    = PORT_OE_OUT;
    PORT_InitStructure.PORT_FUNC  = PORT_FUNC_PORT;
    PORT_InitStructure.PORT_MODE  = PORT_MODE_DIGITAL;
    PORT_InitStructure.PORT_SPEED = PORT_SPEED_SLOW;

    PORT_Init(MDR_PORTA, &PORT_InitStructure);

    /* Erase main memory page MAIN_EEPAGE */
    Address = 0x00000000 + MAIN_EEPAGE * EEPROM_PAGE_SIZE;
    BankSelector = EEPROM_Main_Bank_Select;
    EEPROM_ErasePage(Address, BankSelector);

    /* Check main memory page MAIN_EEPAGE */
    Data = 0xFFFFFFFF;
    for(i = 0; i < EEPROM_PAGE_SIZE; i += 4)
    {
        if (EEPROM_ReadWord(Address + i, BankSelector) != Data)
        {
            EraseErrMM = 1;
            break;
        }
    }

    /* Indicate status of erasing main memory page MAIN_EEPAGE */
    if (EraseErrMM == 0)
    {
        BlinkLED0(3, 100000);
    }
    else
    {
        while(1)
        {
            BlinkLED0(1, 10000);
        }
    }

    /* Fill main memory page MAIN_EEPAGE */
    Address = 0x00000000 + MAIN_EEPAGE * EEPROM_PAGE_SIZE;
    BankSelector = EEPROM_Main_Bank_Select;
    for(i = 0; i < EEPROM_PAGE_SIZE; i += 4)
    {
        Data = Pseudo_Rand(Address + i);
        EEPROM_ProgramWord(Address + i, BankSelector, Data);
    }

    /* Check main memory page MAIN_EEPAGE */
    Address = 0x00000000 + MAIN_EEPAGE * EEPROM_PAGE_SIZE;
    BankSelector = EEPROM_Main_Bank_Select;
    for(i = 0; i < EEPROM_PAGE_SIZE; i +=4 )
    {
        Data = Pseudo_Rand(Address + i);
        if (EEPROM_ReadWord(Address + i, BankSelector) != Data)
        {
            WriteErrMM = 1;
            break;
        }
    }

    /* Indicate status of writing main memory page MAIN_EEPAGE */
    if (EraseErrMM == 0)
    {
        BlinkLED0(4, 200000);
    }
    else
    {
        while(1)
        {
            BlinkLED0(1, 10000);
        }
    }

    /* Erase main memory page MAIN_EEPAGE */
    Address = MAIN_EEPAGE * EEPROM_PAGE_SIZE;
    BankSelector = EEPROM_Main_Bank_Select;
    EEPROM_ErasePage(Address, BankSelector);

    /* Check main memory page MAIN_EEPAGE */
    Data = 0xFFFFFFFF;
    for(i = 0; i < EEPROM_PAGE_SIZE; i += 4)
    {
        if (EEPROM_ReadWord(Address + i, BankSelector) != Data)
        {
            EraseErrMM = 1;
            break;
        }
    }

    /* Indicate status of erasing main memory page MAIN_EEPAGE */
    if (EraseErrMM == 0)
    {
        BlinkLED0(5, 400000);
    }
    else
    {
        while(1)
        {
            BlinkLED0(1, 10000);
        }
    }

    PORT_SetBits(LED0_PORT, LED0_PIN);
    while(1) {}
}

/**
  * @brief  Reports the source file name, the source line number
  *         and expression text (if USE_ASSERT_INFO == 2) where
  *         the assert_param error has occurred.
  * @param  file: pointer to the source file name
  * @param  line: assert_param error line source number
  * @param  expr:
  * @retval None
  */
#if (USE_ASSERT_INFO == 1)
void assert_failed(uint8_t* file, uint32_t line)
{
    /* User can add his own implementation to report the source file name and line number.
       Ex: printf("Wrong parameters value: file %s on line %d\r\n", file, line) */

    /* Infinite loop */
    while(1) {}
}
#elif (USE_ASSERT_INFO == 2)
void assert_failed(uint8_t* file, uint32_t line, const uint8_t* expr)
{
    /* User can add his own implementation to report the source file name, line number and
       expression text.
       Ex: printf("Wrong parameters value (%s): file %s on line %d\r\n", expr, file, line) */

    /* Infinite loop */
    while(1) {}
}
#endif /* USE_ASSERT_INFO */

/** @} */ /* End of group EEPROM_Sector_Operations */

/** @} */ /* End of group __MDR1986BE4_EVAL */

/** @} */ /* End of group __MDR1986BE4_StdPeriph_Examples */

/******************* (C) COPYRIGHT 2024 Milandr ********************************
*
* END OF FILE main.c */

