/**
  ******************************************************************************
  * @file    main.c
  * @author  Milandr Application Team
  * @version V2.1.0
  * @date    27/05/2024
  * @brief   Main program body.
  ******************************************************************************
  * <br><br>
  *
  * THE PRESENT FIRMWARE IS FOR GUIDANCE ONLY. IT AIMS AT PROVIDING CUSTOMERS
  * WITH CODING INFORMATION REGARDING MILANDR'S PRODUCTS IN ORDER TO FACILITATE
  * THE USE AND SAVE TIME. MILANDR SHALL NOT BE HELD LIABLE FOR ANY
  * DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES RESULTING
  * FROM THE CONTENT OF SUCH FIRMWARE AND/OR A USE MADE BY CUSTOMERS OF THE
  * CODING INFORMATION CONTAINED HEREIN IN THEIR PRODUCTS.
  *
  * <h2><center>&copy; COPYRIGHT 2024 Milandr</center></h2>
  */

/* Includes ------------------------------------------------------------------*/
#include "MDR32F9Qx_ssp.h"
#include "MDR32F9Qx_rst_clk.h"
#include "MDR32F9Qx_port.h"

/** @addtogroup __MDR1986BE4_StdPeriph_Examples MDR1986BE4 StdPeriph Examples
  * @{
  */

/** @addtogroup __MDR1986BE4_EVAL MDR1986BE4 Evaluation Board
  * @{
  */

/** @addtogroup SPI_FullDuplex SPI_FullDuplex
  * @{
  */

/* Private define ------------------------------------------------------------*/
#define BufferSize         32

/* Private variables ---------------------------------------------------------*/
SSP_InitTypeDef sSSP;
PORT_InitTypeDef PORT_InitStructure;

uint16_t SrcBuf[BufferSize];
uint16_t DstBuf[BufferSize];

uint8_t TxIdx = 0, RxIdx = 0;

/* Private functions ---------------------------------------------------------*/
void Init_RAM(uint16_t *addr, uint32_t size);
void Zero_RAM(uint16_t *addr, uint32_t size);
uint32_t ps_rand(uint32_t addr);

/**
  * @brief  Main program
  * @param  None
  * @retval None
  */
int main(void)
{
    RST_CLK_DeInit();

    /* Enable HSE clock */
    RST_CLK_HSEconfig(RST_CLK_HSE_ON);
    /* Wait HSE ready */
    while(RST_CLK_HSEstatus() == ERROR);
    /* Select HSE as CPU_C1 source */
    RST_CLK_CPUclkSelectionC1(RST_CLK_CPU_C1srcHSEdiv1);
    /* Select CPU_C1 as CPU_C2 source */
    RST_CLK_CPU_PLLuse(DISABLE);
    /* Select CPU_C2 as CPU_C3 source */
    RST_CLK_CPUclkSelection(RST_CLK_CPUclkCPU_C3);

    SystemCoreClockUpdate();

    /* Enable peripheral clocks */
    RST_CLK_PCLKcmd((RST_CLK_PCLK_RST_CLK | RST_CLK_PCLK_SSP1 |
                    RST_CLK_PCLK_PORTA), ENABLE);

    /* Reset PORTA settings */
    PORT_DeInit(MDR_PORTA);

    /*Init structure*/
    PORT_StructInit(&PORT_InitStructure);

    /* Configure SSP1 pins: FSS, CLK, RXD, TXD */
    /* Configure PORTA pins 12, 13, 15 */
    PORT_InitStructure.PORT_Pin   = (PORT_Pin_12 | PORT_Pin_13 | PORT_Pin_15);
    PORT_InitStructure.PORT_FUNC  = PORT_FUNC_MAIN;
    PORT_InitStructure.PORT_MODE  = PORT_MODE_DIGITAL;
    PORT_InitStructure.PORT_SPEED = PORT_SPEED_FAST;
    PORT_InitStructure.PORT_OE    = PORT_OE_OUT;
    PORT_Init(MDR_PORTA, &PORT_InitStructure);

    PORT_InitStructure.PORT_Pin       = (PORT_Pin_14);
    PORT_InitStructure.PORT_OE        = PORT_OE_IN;
    PORT_InitStructure.PORT_PULL_DOWN = PORT_PULL_DOWN_ON;
    PORT_Init(MDR_PORTA, &PORT_InitStructure);

    /* Reset all SSP settings */
    SSP_DeInit();

    SSP_BRGInit(SSP_HCLKdiv16);

    /* SSP1 MASTER configuration */
    SSP_StructInit (&sSSP);

    sSSP.SSP_SCR        = 0x10;
    sSSP.SSP_CPSDVSR    = 2;
    sSSP.SSP_Mode       = SSP_ModeMaster;
    sSSP.SSP_WordLength = SSP_WordLength16b;
    sSSP.SSP_SPH        = SSP_SPH_1Edge;
    sSSP.SSP_SPO        = SSP_SPO_Low;
    sSSP.SSP_FRF        = SSP_FRF_SPI_Motorola;
    sSSP.SSP_HardwareFlowControl = SSP_HardwareFlowControl_SSE;
    SSP_Init(&sSSP);

    /* Enable SSP1 */
    SSP_Cmd(ENABLE);

    /* Init RAM */
    Zero_RAM(DstBuf, BufferSize);
    Init_RAM(SrcBuf, BufferSize);

    while(1)
    {
        /* Transfer procedure */
        while(TxIdx < BufferSize)
        {
            /* Wait for SPI1 Tx buffer not full */
            while(SSP_GetFlagStatus(SSP_FLAG_TNF) == RESET);

            /* Send SPI1 data */
            SSP_SendData(SrcBuf[TxIdx++]);

            /* Wait for SPI1 data reception */
            while(SSP_GetFlagStatus(SSP_FLAG_RNE) == RESET);

            /* Read SPI1 received data */
            DstBuf[RxIdx++] = SSP_ReceiveData();
        }
    }
}

/**
 * @brief  Initialisation memory pseudo random value
 * @param  addr - memory address
 * @param  size - memory size
 * @retval None
 */
void Init_RAM(uint16_t *addr, uint32_t size)
{
    uint32_t i;

    for(i = 0; i < size; i++)
    {
        *addr = ps_rand((uint32_t)addr + i * 4);
        addr++;
    }
}

/**
 * @brief  Zeroing function memory.
 * @param  addr - memory address
 * @param  size - memory size
 * @retval None
 */
void Zero_RAM(uint16_t *addr, uint32_t size)
{
    uint32_t i;
    for(i = 0; i < size; i++)
    {
        addr[i] = 0;
    }
}

/**
 * @brief  Generation random values
 * @param  addr:
 * @retval random value
 */
uint32_t ps_rand(uint32_t addr)
{
    uint32_t hash = 0;
    uint32_t i;
    char *key = (char *) &addr;

    for(i = 0; i < 4; i++)
    {
        hash += key[i];
        hash += (hash << 10);
        hash ^= (hash >> 6);
    }

    for(i = 0; i < 256; i++)
    {
        hash += (hash << 10);
        hash ^= (hash >> 6);
    }

    hash += (hash << 3);
    hash ^= (hash >> 11);
    hash += (hash << 15);

    return hash;
}

/**
  * @brief  Reports the source file name, the source line number
  *         and expression text (if USE_ASSERT_INFO == 2) where
  *         the assert_param error has occurred.
  * @param  file: pointer to the source file name
  * @param  line: assert_param error line source number
  * @param  expr:
  * @retval None
  */
#if (USE_ASSERT_INFO == 1)
void assert_failed(uint8_t* file, uint32_t line)
{
    /* User can add his own implementation to report the source file name and line number.
       Ex: printf("Wrong parameters value: file %s on line %d\r\n", file, line) */

    /* Infinite loop */
    while(1) {}
}
#elif (USE_ASSERT_INFO == 2)
void assert_failed(uint8_t* file, uint32_t line, const uint8_t* expr)
{
    /* User can add his own implementation to report the source file name, line number and
       expression text.
       Ex: printf("Wrong parameters value (%s): file %s on line %d\r\n", expr, file, line) */

    /* Infinite loop */
    while(1) {}
}
#endif /* USE_ASSERT_INFO */

/** @} */ /* End of group SPI_FullDuplex */

/** @} */ /* End of group __MDR1986BE4_EVAL */

/** @} */ /* End of group __MDR1986BE4_StdPeriph_Examples */

/******************* (C) COPYRIGHT 2024 Milandr *******************************/

/* END OF FILE main.c */

