
/* Includes ------------------------------------------------------------------*/
#include "MDR32VF0xI_rst_clk.h"
#include "MDR32VF0xI_utils.h"
#include "MDR32VF0xI_port.h"
#include "MDR32VF0xI_ssp.h"

/* Private functions prototypes ----------------------------------------------*/
void SPI_Config(void);
void SPI_PinsConfig(void);

/* Private variables ---------------------------------------------------------*/
volatile uint32_t d0, d1, d2, d3; // Watch these variables during debug.

/* Private functions ---------------------------------------------------------*/
/**
 * @brief  Main program.
 * @param  None.
 * @return None.
 */
int main(void)
{
    // W25Q64FVAIQ is connected to MDR1206FI demoboard (XP12 / SSP1) same way as described in https://github.com/Tamakichi/Arduino-W25Q64/blob/master/img/Schematic3.jpg
    // There are pullup resistors on nCS and RXD on demoboard (see https://ic.milandr.ru/products/programmno_otladochnye_sredstva/otladochnye_komplekty/otladochnyy-komplekt-dlya-mikroskhemy-mdr1206fi/)

    RST_CLK_DeInit();
    RST_CLK_HCLK_ClkSelection(RST_CLK_CPU_HCLK_CLK_SRC_HSI);

    SPI_Config();
    SPI_PinsConfig();
    SSP_Cmd(MDR_SSP1, ENABLE);

    // Read W25Q64FV JEDEC ID command.
    // See W25Q64FV datasheet to learn more about commands.

    SSP_SendData(MDR_SSP1, 0x9F); // NOTE: All 4 writes MUST be executed simultaneously (so nCS will be low during all 4 transactions), so don't execute them in debug step-by-step!
    SSP_SendData(MDR_SSP1, 0x0); // Dummy data to form CLK (MDR1206FI is master).
    SSP_SendData(MDR_SSP1, 0x0); // Dummy data to form CLK.
    SSP_SendData(MDR_SSP1, 0x0); // Dummy data to form CLK.

    while (!SSP_GetFlagStatus(MDR_SSP1, SSP_FLAG_TFE)) {}
    while (SSP_GetFlagStatus(MDR_SSP1, SSP_FLAG_BSY)) {}

    // NOTE: SSP DR reg is FIFO, so reading this reg during debug will break program.
    d0 = SSP_ReceiveData(MDR_SSP1); // Should be 0xFF (because RXD is pulled up on demoboard) - read from FIFO trash data which were send on 0x9F command.
    d1 = SSP_ReceiveData(MDR_SSP1); // Should be 0xEF (Winbond Serial Flash)
    d2 = SSP_ReceiveData(MDR_SSP1); // Should be 0x40 (W25Q64FV Memory Type (ID15-ID8))
    d3 = SSP_ReceiveData(MDR_SSP1); // Should be 0x17 (W25Q64FV Capacity (ID7-ID0))

    while (1) {}
}

/**
 * @brief  SPI configuration.
 * @param  None.
 * @return None.
 */
void SPI_Config(void)
{
    static const SSP_InitTypeDef SPI_W25 = {
        .SSP_CPSDVSR       = 16,
        .SSP_SCR           = 0,
        .SSP_Mode          = SSP_MODE_MASTER,
        .SSP_WordLength    = SSP_WORD_LENGTH_8BIT,
        .SSP_ClockPhase    = SSP_CLOCK_PHASE_2EDGE,
        .SSP_ClockPolarity = SSP_CLOCK_POLARITY_HIGH, // Setup MODE 3 (1, 1) as MCU holds nCS in LOW in this mode during transfer of multiple data.
        .SSP_FrameFormat   = SSP_FRAME_FORMAT_SPI,
        .SSP_LoopbackMode  = DISABLE,
    };

    RST_CLK_PER1_C2_ClkSelection(RST_CLK_PER1_C2_CLK_SRC_CPU_C1);
    RST_CLK_PER1_C2_Cmd(RST_CLK_PER1_C2_SSP1, ENABLE);

    RST_CLK_PCLKCmd(RST_CLK_PCLK_SSP1, ENABLE);
    SSP_DeInit(MDR_SSP1);
    SSP_Init(MDR_SSP1, &SPI_W25);
}

/**
 * @brief  SPI pins configuration.
 * @param  None.
 * @return None.
 */
void SPI_PinsConfig(void)
{
    static const PORT_InitTypeDef SPI_Pins = {
        .PORT_Pin       = PORT_PIN_12 | PORT_PIN_13 | PORT_PIN_14 | PORT_PIN_15,
        .PORT_Mode      = PORT_MODE_DIGITAL,
        .PORT_Function  = PORT_FUNCTION_MAIN,
        .PORT_Power     = PORT_POWER_NOMINAL_UPTO_2mA,
#if !defined(USE_MDR32F02_REV_1X)
        .PORT_PullUp = PORT_PULL_UP_OFF,
#endif
        .PORT_PullDown = PORT_PULL_DOWN_OFF,
    };

    RST_CLK_PCLKCmd(RST_CLK_PCLK_PORTA, ENABLE);
    PORT_Init(MDR_PORTA, &SPI_Pins);
}

